# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import timedelta

import mongolock
from django.conf import settings
from django.db import transaction

from travel.rasp.library.python.common23.date import environment
from common.utils.date import MSK_TZ
from common.utils.lock import lock
from travel.rasp.library.python.common23.logging.scripts import script_context
from common.dynamic_settings.core import DynamicSetting
from common.dynamic_settings.default import conf
from travel.rasp.suburban_tasks.suburban_tasks.cpy_pst import get_script_logger
from travel.rasp.suburban_tasks.suburban_tasks.models import Update, Change_SPEC_BUF, Change_STRAINS_BUF, Change_STRAINSVAR_BUF, Change_SRASPRP_BUF, \
    Change_SCALENDAR_BUF, Change_SDOCS_BUF, Current_STRAINS, Current_STRAINSVAR, Current_SRASPRP, Current_SCALENDAR
from travel.rasp.suburban_tasks.suburban_tasks.rzd_utils import get_changes_spec_buf_rows, get_train_changes_related_data, \
    use_rzd_db_manager, GVC_PROC_DATE_FORMAT
from travel.rasp.suburban_tasks.suburban_tasks.utils import build_spec_buf_key


conf.register_settings(
    RZD_CHANGES_MAX_HOURS_TO_FETCH=DynamicSetting(0, cache_time=60,
                                                  description='За сколько часов максимум пытаемся выгрузить расписания '
                                                              'за один запуск скрипта'),
)


log = get_script_logger()


class HasNoFullUpdate(Exception):
    pass


class Bad_SPEC_BUF_KOP(Exception):
    pass


def format_diffs(diffs):
    return u';'.join(u'{field}: "{from}" -> "{to}"'.format(**diff) for diff in diffs)


def action_update(args):
    update = load_changes_update()
    apply_changes_update(update)


def action_load_only(args):
    load_changes_update()


def action_apply_last(args):
    try:
        update = Update.objects.filter(action_type=Update.CHANGES_UPDATE).order_by('-updated_at')[0]
    except IndexError:
        log.error(u'Не нашли загруженного полного обновления')
        return
    else:
        apply_changes_update(update)


def action_apply_by_id(args):
    update = Update.objects.get(pk=args.update_id, action_type=Update.CHANGES_UPDATE)
    log.info(u'Применяем обновление %s %s', update.id, update.updated_at)
    apply_changes_update(update)


@transaction.atomic(using=settings.MYSQL_RZD_DB_ALIAS)
@use_rzd_db_manager
def load_changes_update():
    update = load_new_spec_bufs()
    load_changes_for_update(update)

    return update


def drop_spec_buf_microseconds(spec_buf_rows):
    # https://st.yandex-team.ru/RASPADMIN-1041#1456483934000
    for row in spec_buf_rows:
        row['DATE_GVC'] = row['DATE_GVC'].replace(microsecond=0)


def get_load_new_spec_bufs_time_range(last_update):
    query_from_dt = last_update.last_gvc_date
    query_to_dt = environment.now_aware().astimezone(MSK_TZ).replace(tzinfo=None)

    max_hours = conf.RZD_CHANGES_MAX_HOURS_TO_FETCH
    if max_hours != 0 and query_to_dt - query_from_dt > timedelta(hours=max_hours):
        query_to_dt = query_from_dt + timedelta(hours=max_hours)

    return query_from_dt, query_to_dt


def load_new_spec_bufs():
    if not Update.objects.filter(action_type=Update.FULL_UPDATE).exists():
        log.error(u'Сначала нужно импортировать расписания РЖД полностью')
        raise HasNoFullUpdate(u'Сначала нужно импортировать расписания РЖД полностью')

    last_update = Update.objects.all().order_by('-updated_at')[0]
    query_from_dt, query_to_dt = get_load_new_spec_bufs_time_range(last_update)

    update = Update(action_type=Update.CHANGES_UPDATE, query_from_dt=query_from_dt, query_to_dt=query_to_dt)

    log.info(u'Запрашиваем изменения %s - %s', query_from_dt.strftime(GVC_PROC_DATE_FORMAT),
             query_to_dt.strftime(GVC_PROC_DATE_FORMAT))
    spec_buf_rows = get_changes_spec_buf_rows(query_from_dt, query_to_dt)
    log.info(u'Получено %s идентификаторов изменений', len(spec_buf_rows))
    spec_buf_rows = filter_spec_buf_rows(spec_buf_rows, last_update)
    log.info(u'Получено %s новых идентификаторов изменений', len(spec_buf_rows))

    drop_spec_buf_microseconds(spec_buf_rows)

    if spec_buf_rows:
        update.last_gvc_date = spec_buf_rows[-1]['DATE_GVC']
        update.is_fake_gvc_date = False
    else:
        update.last_gvc_date = last_update.last_gvc_date
        update.is_fake_gvc_date = last_update.is_fake_gvc_date

    update.save()

    Change_SPEC_BUF.objects.bulk_create([Change_SPEC_BUF(update=update, **r) for r in spec_buf_rows])

    return update


def filter_spec_buf_rows(spec_buf_rows, last_update):
    if not spec_buf_rows:
        return spec_buf_rows

    oldest_spec_buf_row = spec_buf_rows[0]

    existed_keys = [sb.key for sb in Change_SPEC_BUF.objects.filter(DATE_GVC__gte=oldest_spec_buf_row['DATE_GVC'])]

    if last_update.action_type == Update.FULL_UPDATE:
        last_full_update = last_update
    else:
        last_full_update = Update.objects.filter(updated_at__lte=last_update.updated_at,
                                                 action_type=Update.FULL_UPDATE).order_by('-updated_at')[0]

    bottom_date_exclusive_limit = last_full_update.last_gvc_date

    new_spec_buf_rows = []
    for spec_buf_row in spec_buf_rows:
        if spec_buf_row['DATE_GVC'].replace(microsecond=0) <= bottom_date_exclusive_limit:
            continue

        if build_spec_buf_key(spec_buf_row) in existed_keys:
            continue

        new_spec_buf_rows.append(spec_buf_row)

    return new_spec_buf_rows


def load_changes_for_update(update):
    spec_buf_keys = [sb.key for sb in Change_SPEC_BUF.objects.filter(update=update)]
    train_ids = list(set(Change_SPEC_BUF.objects.filter(update=update).values_list('IDTR', flat=True)))

    # расширяем интервал, чтобы точно зацепить все изменения
    # https://st.yandex-team.ru/RASPFRONT-6207#1538496242000
    dt_to = update.query_to_dt + timedelta(hours=2)

    data = get_train_changes_related_data(train_ids, update.query_from_dt, dt_to)

    model_map = {
        'STRAINS': Change_STRAINS_BUF,
        'STRAINSVAR': Change_STRAINSVAR_BUF,
        'SRASPRP': Change_SRASPRP_BUF,
        'SCALENDAR': Change_SCALENDAR_BUF,
        'SDOCS': Change_SDOCS_BUF,
    }

    for rows in data.values():
        drop_spec_buf_microseconds(rows)

    def filterout_excess_buf_rows(buf_rows):
        return [r for r in buf_rows if build_spec_buf_key(r) in spec_buf_keys]

    for tbl_name, model in model_map.items():
        objs = []
        for rowdict in filterout_excess_buf_rows(data[tbl_name]):
            objs.append(model(update=update, **rowdict))

        model.objects.bulk_create(objs)

    for tbl_name, model in model_map.items():
        log.info(u'Загрузили %s %s', tbl_name, model.objects.filter(update=update).count())

    log.info(u'Изменения сохранены в нашу базу')


@transaction.atomic(using=settings.MYSQL_RZD_DB_ALIAS)
def apply_changes_update(update):
    log.info(u'Применяем обновление за %s', update.updated_at)

    spec_bufs = Change_SPEC_BUF.objects.filter(update=update).order_by('DATE_GVC')
    for spec_buf in spec_bufs:
        log.info(u'Применяем %s', spec_buf)
        apply_strains(spec_buf)
        apply_strainsvar(spec_buf)
        apply_srasprp(spec_buf)
        apply_scalendar(spec_buf)

    log.info(u'Обновление %s успешно применено', update.updated_at)


def apply_strains(spec_buf):
    for change_strain in Change_STRAINS_BUF.get_by_spec_buf(spec_buf):
        try:
            current_strain = Current_STRAINS.objects.get(IDTR=change_strain.IDTR)
        except Current_STRAINS.DoesNotExist:
            current_strain = None

        if spec_buf.KOP in (Change_SPEC_BUF.KOP_INSERT_MODE, Change_SPEC_BUF.KOP_UPDATE_MODE):
            if current_strain:
                diffs = current_strain.update_from_update_object(change_strain)
                if diffs:
                    current_strain.save()
                    log.info(u'Обновили STRAINS IDTR=%s. %s. %s', current_strain.IDTR, format_diffs(diffs), spec_buf)
                else:
                    log.info(u'Уже обновлено или создано STRAINS IDTR=%s. %s', current_strain.IDTR, spec_buf)
            else:
                new_strain = Current_STRAINS.create_from_update_object(change_strain)
                log.info(u'Создали STRAINS IDTR=%s. %s', new_strain.IDTR, spec_buf)
        elif spec_buf.KOP == Change_SPEC_BUF.KOP_DELETE_MODE:
            if current_strain:
                log.info(u'Удалили STRAINS IDTR=%s. %s', change_strain.IDTR, spec_buf)
                current_strain.delete()
            else:
                log.info(u'Уже удалено STRAINS IDTR=%s. %s', change_strain.IDTR, spec_buf)
        else:
            log.error(u'Неверный KOP %s', spec_buf.KOP)
            raise Bad_SPEC_BUF_KOP(u'Неверный KOP {}'.format(spec_buf.KOP))


def apply_strainsvar(spec_buf):
    for change_strainsvar in Change_STRAINSVAR_BUF.get_by_spec_buf(spec_buf):
        try:
            current_strainsvar = Current_STRAINSVAR.objects.get(IDR=change_strainsvar.IDR)
        except Current_STRAINSVAR.DoesNotExist:
            current_strainsvar = None

        if spec_buf.KOP in (Change_SPEC_BUF.KOP_INSERT_MODE, Change_SPEC_BUF.KOP_UPDATE_MODE):
            if current_strainsvar:
                diffs = current_strainsvar.update_from_update_object(change_strainsvar)
                if diffs:
                    current_strainsvar.save()
                    log.info(u'Обновили STRAINSVAR IDR=%s. %s. %s', current_strainsvar.IDR, format_diffs(diffs), spec_buf)
                else:
                    log.info(u'Уже обновлено или создано STRAINSVAR IDR=%s. %s', current_strainsvar.IDR, spec_buf)
            else:
                new_strainvar = Current_STRAINSVAR.create_from_update_object(change_strainsvar)
                log.info(u'Создали STRAINSVAR IDR=%s. %s', new_strainvar.IDR, spec_buf)
        elif spec_buf.KOP == Change_SPEC_BUF.KOP_DELETE_MODE:
            if current_strainsvar:
                log.info(u'Удалили STRAINSVAR IDR=%s. %s', change_strainsvar.IDR, spec_buf)
                current_strainsvar.delete()
            else:
                log.info(u'Уже удалено STRAINSVAR IDR=%s. %s', change_strainsvar.IDR, spec_buf)

        else:
            log.error(u'Неверный KOP %s', spec_buf.KOP)
            raise Bad_SPEC_BUF_KOP(u'Неверный KOP {}'.format(spec_buf.KOP))


def apply_srasprp(spec_buf):
    strainsvar_ids = set(Change_SRASPRP_BUF.get_by_spec_buf(spec_buf).values_list('IDR', flat=True))
    for IDR in strainsvar_ids:
        prev_objects = list(Current_SRASPRP.objects.filter(IDR=IDR).order_by('IDTR', 'IDR', 'SEQ', 'IDRP'))
        new_objects = [
            Current_SRASPRP.build_from_update_object(o)
            for o in Change_SRASPRP_BUF.get_by_spec_buf(spec_buf).filter(IDR=IDR).order_by('IDTR', 'IDR', 'SEQ', 'IDRP')
        ]

        has_difference = len(prev_objects) != len(new_objects) or any(
            Current_SRASPRP.diff_by_base_fields(obj1, obj2) for obj1, obj2 in zip(prev_objects, new_objects)
        )

        if spec_buf.KOP in (Change_SPEC_BUF.KOP_INSERT_MODE, Change_SPEC_BUF.KOP_UPDATE_MODE):
            if not has_difference:
                log.info(u'Уже обновлено или создано SRASPRP IDR=%s. %s', IDR, spec_buf)
            else:
                if prev_objects:
                    Current_SRASPRP.objects.filter(IDR=IDR).delete()
                    Current_SRASPRP.objects.bulk_create(new_objects)
                    log.info(u'Обновили SRASPRP IDR=%s, prev_count=%s, new_count=%s. %s',
                             IDR, len(prev_objects), len(new_objects), spec_buf)
                else:
                    Current_SRASPRP.objects.bulk_create(new_objects)
                    log.info(u'Создали SRASPRP IDR=%s, new_count=%s. %s',
                             IDR, len(new_objects), spec_buf)
        elif spec_buf.KOP == Change_SPEC_BUF.KOP_DELETE_MODE:
            if not prev_objects:
                log.info(u'Уже удалено SRASPRP IDR=%s. %s', IDR, spec_buf)
            else:
                Current_SRASPRP.objects.filter(IDR=IDR).delete()
                log.info(u'Удалили SRASPRP IDR=%s. %s', IDR, spec_buf)

        else:
            log.error(u'Неверный KOP %s', spec_buf.KOP)
            raise Bad_SPEC_BUF_KOP(u'Неверный KOP {}'.format(spec_buf.KOP))


def apply_scalendar(spec_buf):
    for change_scalendar in Change_SCALENDAR_BUF.get_by_spec_buf(spec_buf):
        try:
            current_scalendar = Current_SCALENDAR.objects.get(IDTR=change_scalendar.IDTR,
                                                              CDATE=change_scalendar.CDATE)
        except Current_SCALENDAR.DoesNotExist:
            current_scalendar = None

        if spec_buf.KOP in (Change_SPEC_BUF.KOP_INSERT_MODE, Change_SPEC_BUF.KOP_UPDATE_MODE):
            if current_scalendar:
                diffs = current_scalendar.update_from_update_object(change_scalendar)
                if diffs:
                    current_scalendar.save()
                    log.info(u'Обновили SCALENDAR IDTR=%s, CDATE=%s. %s. %s',
                             current_scalendar.IDTR, current_scalendar.CDATE,
                             format_diffs(diffs), spec_buf)
                else:
                    log.info(u'Уже обновлено или создано SCALENDAR IDTR=%s, CDATE=%s. %s', current_scalendar.IDTR,
                             current_scalendar.CDATE, spec_buf)
            else:
                new_scalendar = Current_SCALENDAR.create_from_update_object(change_scalendar)
                log.info(u'Создали SCALENDAR IDTR=%s, CDATE=%s. %s', new_scalendar.IDTR, new_scalendar.CDATE, spec_buf)
        elif spec_buf.KOP == Change_SPEC_BUF.KOP_DELETE_MODE:
            if current_scalendar:
                log.info(u'Удалили SCALENDAR IDTR=%s, CDATE=%s. %s', current_scalendar.IDTR, current_scalendar.CDATE,
                         spec_buf)
                current_scalendar.delete()
            else:
                log.info(u'Уже удалено SCALENDAR IDTR=%s, CDATE=%s. %s', change_scalendar.IDTR,
                         change_scalendar.CDATE, spec_buf)

        else:
            log.error(u'Неверный KOP %s', spec_buf.KOP)
            raise Bad_SPEC_BUF_KOP(u'Неверный KOP {}'.format(spec_buf.KOP))


def run(action, args_obj=None):
    try:
        with lock('rzd_trains_data_update', database_name=settings.SUBURBAN_EVENTS_DATABASE_NAME),\
             script_context('update_changes', report_progress=False):
            from travel.rasp.suburban_tasks.suburban_tasks.cpy_pst import print_log_to_stdout

            if args_obj and args_obj.verbose:
                print_log_to_stdout('suburban_tasks')

            action_func = globals()['action_{}'.format(action.replace('-', '_'))]
            action_func(args_obj)
    except mongolock.MongoLockLocked as ex:
        log.debug('Can not get lock: %s', repr(ex))


def main():
    import argparse

    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--verbose', action='store_true')

    subparsers = parser.add_subparsers(dest='action')
    for simple_action in ('update', 'load-only', 'apply-last'):
        subparsers.add_parser(simple_action)

    apply_by_id_parser = subparsers.add_parser('apply-by-id')
    apply_by_id_parser.add_argument('update_id', type=int)

    args = parser.parse_args()
    run(args.action, args)


if __name__ == '__main__':
    main()
