# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import timedelta

import mongolock

from django.conf import settings
from django.db import transaction

from travel.rasp.library.python.common23.date import environment
from common.utils.date import MSK_TZ
from common.utils.lock import lock
from travel.rasp.library.python.common23.logging.scripts import script_context
from travel.rasp.suburban_tasks.suburban_tasks.rzd_utils import (
    get_stations, get_gdpprbase_regions, get_gdpprbase_strains_for_region, get_trains_related_data, get_changes_spec_buf_rows, use_rzd_db_manager, GVC_PROC_DATE_FORMAT
)
from travel.rasp.suburban_tasks.suburban_tasks import models as rzd_models
from travel.rasp.suburban_tasks.suburban_tasks.models import IC00_STAN, Full_STRAINS, Full_STRAINSVAR, Full_SRASPRP, Full_SCALENDAR, Full_SDOCS, Update
from travel.rasp.suburban_tasks.suburban_tasks.cpy_pst import get_script_logger, model_iter


log = get_script_logger()

LAST_CHANGES_PAST_DATE = 1  # на сколько суток запросить в прошлое, чтобы найти последние изменения
MIN_DAYS_FROM_LAST_FULL_UPDATE = 31 * 6
# Судя по договору база обновляется раз в сутки, поэтому если изменений за последнее время не было
# Берем дату на 12 часов назад
DEFAULT_LAST_GVC_DATE_OFFSET = 0.5


class HasFreshImportException(Exception):
    pass


@use_rzd_db_manager
def action_update():
    update = update_full_update_tables()
    apply_full_update(update)


@use_rzd_db_manager
def action_load_only():
    update_full_update_tables()


def action_apply_last():
    try:
        update = Update.objects.filter(action_type=Update.FULL_UPDATE).order_by('-updated_at')[0]
    except IndexError:
        log.error(u'Не нашли загруженного полного обновления')
        return
    else:
        apply_full_update(update)


@use_rzd_db_manager
def action_reload_stations():
    reload_stations()


@transaction.atomic(using=settings.MYSQL_RZD_DB_ALIAS)
def update_full_update_tables():
    # Логируем в начале, на тот случай, если изменения появятся в процессе
    last_full_updates = Update.objects.filter(action_type=Update.FULL_UPDATE).order_by('-updated_at')
    if last_full_updates.exists():
        last_full_update = last_full_updates[0]
        days_from_last_update = (environment.today() - last_full_update.updated_at.date()).total_seconds() / 3600 / 24
        if days_from_last_update < MIN_DAYS_FROM_LAST_FULL_UPDATE:
            log.error(u'Нельзя обновлять базу, т.к. еще не прошло %s дней', MIN_DAYS_FROM_LAST_FULL_UPDATE)
            raise HasFreshImportException(u'Нельзя обновлять базу, т.к. еще не прошло {} дней'
                                          .format(MIN_DAYS_FROM_LAST_FULL_UPDATE))

    update = Update(action_type=Update.FULL_UPDATE)
    set_last_gvc_date(update)
    update.save()

    update_rzd_tables(update)

    return update


@transaction.atomic(using=settings.MYSQL_RZD_DB_ALIAS)
def apply_full_update(update):
    log.info(u'Применяем обновление за %s', update.updated_at)

    rzd_tables = ('STRAINS', 'STRAINSVAR', 'SRASPRP', 'SCALENDAR', 'SDOCS')

    for rzd_name in rzd_tables:
        full_update_model = getattr(rzd_models, 'Full_{}'.format(rzd_name))
        current_model = getattr(rzd_models, 'Current_{}'.format(rzd_name))

        log.info(u'Удаляем старые данные из %s', current_model)
        current_model.objects.all().delete()

        log.info(u'Заливаем данные из %s в %s', full_update_model, current_model)
        for update_objs in model_iter(full_update_model.objects.filter(update=update), chunksize=10000, in_chunks=True):
            current_objects = [current_model.build_from_update_object(update_obj)
                               for update_obj in update_objs]

            current_model.objects.bulk_create(current_objects)

        log.info(u'Загрузили %s объектов %s', current_model.objects.count(), current_model)


def set_last_gvc_date(update):
    log.info(u'Ищем дату последнего изменения')
    msk_now = environment.now_aware().astimezone(MSK_TZ).replace(tzinfo=None)
    query_from = msk_now - timedelta(LAST_CHANGES_PAST_DATE)
    query_to = msk_now

    log.info(u'Ищем изменения в диапазоне %s %s', query_from.strftime(GVC_PROC_DATE_FORMAT),
             query_to.strftime(GVC_PROC_DATE_FORMAT))
    rows = get_changes_spec_buf_rows(query_from, query_to)

    if rows:
        update.last_gvc_date = rows[-1]['DATE_GVC']
        update.is_fake_gvc_date = False
        log.info(u'Нашли дату последнего изменения %s', update.last_gvc_date)
    else:
        update.last_gvc_date = msk_now - timedelta(DEFAULT_LAST_GVC_DATE_OFFSET)
        update.is_fake_gvc_date = True
        log.info(u'Не нашли дату последнего изменения ставим фековую %s', update.last_gvc_date)


def update_rzd_tables(update):
    load_strains(update)
    load_train_related_data(update)


def load_strains(update):
    log.info(u'Заливаем поезда')

    for region_dict in get_gdpprbase_regions():
        trains = []
        log.info(u'Получаем поезда для региона %s %s',
                 unicode(region_dict['CREG']).strip(), unicode(region_dict['NNAPR']).strip())
        for train_dict in get_gdpprbase_strains_for_region(region_dict['CREG'], environment.today()):
            trains.append(Full_STRAINS(update=update, **train_dict))

        Full_STRAINS.objects.bulk_create(trains)

    log.info(u'Загружено %s поездов', Full_STRAINS.objects.filter(update=update).count())


def load_train_related_data(update):
    log.info(u'Загружаем данные связанные с поедами')

    model_map = {
        'STRAINSVAR': Full_STRAINSVAR,
        'SRASPRP': Full_SRASPRP,
        'SCALENDAR': Full_SCALENDAR,
        'SDOCS': Full_SDOCS,
    }

    train_ids = list(Full_STRAINS.objects.filter(update=update).values_list('IDTR', flat=True))

    for data in get_trains_related_data(train_ids):
        for tbl_name, model in model_map.items():
            objs = []
            for rowdict in data[tbl_name]:
                objs.append(model(update=update, **rowdict))

            model.objects.bulk_create(objs)

    for tbl_name, model in model_map.items():
        log.info(u'Загрузили %s %s', tbl_name, model.objects.filter(update=update).count())


def reload_stations():
    log.info(u'Перезаливаем станции')
    IC00_STAN.objects.all().delete()
    stations = []
    for station_dict in get_stations():
        stations.append(IC00_STAN(**station_dict))

    IC00_STAN.objects.bulk_create(stations)
    log.info(u'Загружено %s станций', IC00_STAN.objects.count())


def run(action=None):
    try:
        with lock('rzd_trains_data_update', database_name=settings.SUBURBAN_EVENTS_DATABASE_NAME),\
             script_context('update_full', report_progress=False):
            import argparse
            from travel.rasp.suburban_tasks.suburban_tasks.cpy_pst import print_log_to_stdout

            parser = argparse.ArgumentParser()
            parser.add_argument('action', choices=('update', 'load-only', 'apply-last', 'reload-stations'))
            parser.add_argument('-v', '--verbose', action='store_true')

            args = parser.parse_args()

            if args.verbose:
                print_log_to_stdout('suburban_tasks')

            action = action or args.action
            action_func = globals()['action_{}'.format(action.replace('-', '_'))]
            action_func()
    except mongolock.MongoLockLocked as ex:
        log.debug('Can not get lock: %s', repr(ex))


if __name__ == '__main__':
    run()
