# -*- coding: utf-8 -*-

from datetime import timedelta
from functools import wraps
from itertools import chain

from django.conf import settings
from django.utils import translation

from common.models.geo import (Settlement, ExternalDirection, ExternalDirectionMarker,
                               Station, SuburbanZone)
from common.models_utils.i18n import RouteLTitle
from travel.rasp.library.python.common23.date import environment
from common.utils.ya import detect_lang
from common.utils.httpresponses import jsonp_wrap, jsonp_response
from route_search.models import ZNodeRoute2
from route_search.shortcuts import find
from travel.rasp.suburban_widget.suburban_widget.views.helpers import suburbanwizard
from travel.rasp.suburban_widget.suburban_widget.views.helpers.jinja import render_to_response


def detect_lang_only_by_cookie(func):
    @wraps(func)
    def wrapper(request, *args, **kwargs):
        lang = detect_lang(request, supported_langs=settings.FRONTEND_LANGUAGES)

        if lang:
            # Если не определился оставляем какой был
            translation.activate(lang)

        return func(request, *args, **kwargs)

    return wrapper


def get_station_from_request(request, direction):
    rasp_id = request.GET['rasp_%s' % direction]

    queryset = Station.objects.filter(hidden=False)

    return queryset.get(id=rasp_id)


def int_or_default(value, default):
    try:
        return int(value)

    except (ValueError, TypeError):
        return default


def next_trains(request):
    try:
        station_from = get_station_from_request(request, 'from')
        station_to = get_station_from_request(request, 'to')

    except (KeyError, ValueError, Station.DoesNotExist):
        # Пробуем взять из колдунщика
        try:
            station_from_id, station_to_id = suburbanwizard.search_direction(request)

            station_from = Station.objects.get(id=station_from_id)
            station_to = Station.objects.get(id=station_to_id)

        except KeyError:
            return jsonp_wrap(request, None)

    if 'trough_service_exists' in request.GET:
        trough_service_exists = ZNodeRoute2.objects.filter(
            station_from=station_from,
            station_to=station_to,
            route__t_type__id=6
        ).exists()

        return jsonp_wrap(request, trough_service_exists)

    n = int_or_default(request.GET.get('n'), 5)

    minutes_forward = int_or_default(request.GET.get('minutes_forward'), 0)
    minutes_backward = int_or_default(request.GET.get('minutes_backward'), 0)

    # Вычисление времен и дат для выборки ближайших рейсов
    now_aware = environment.now_aware()

    COOKIE_KEY = 'next_trains_counter'

    try:
        show_counter = int(request.COOKIES[COOKIE_KEY])
    except (KeyError, ValueError):
        show_counter = 0

    show_settings = show_counter < 5

    res = {
        'from': {
            'id': station_from.id,
            'name': station_from.get_popular_title(suburban=True),
        },
        'to': {
            'id': station_to.id,
            'name': station_to.get_popular_title(suburban=True),
        },
        'routes_forward': next_suburban_trains(station_from, station_to, n, now_aware + timedelta(minutes=minutes_forward)),
        'routes_backward': next_suburban_trains(station_to, station_from, n, now_aware + timedelta(minutes=minutes_backward)),
        'show_settings': show_settings,
    }

    response = jsonp_wrap(request, res)

    show_counter += 1

    response.set_cookie(COOKIE_KEY, show_counter)

    return response


def next_suburban_trains(station_from, station_to, n, dt_from):
    routes = []

    segments = list(find_next_suburban_trains(station_from, station_to, n, dt_from))

    RouteLTitle.fetch([s.thread.L_title for s in segments if s.thread])

    for segment in segments:
        route = {
            'number': segment.thread.number,
            'title': segment.thread.L_title(short=True),
            'express': segment.thread.express_type == 'express',
            'departure': segment.departure.strftime('%H:%M'),
            'arrival': segment.arrival.strftime('%H:%M'),
            'uid': segment.thread.uid,
        }

        routes.append(route)

    return routes


def find_next_suburban_trains(point_from, point_to, n, dt_from):
    count = 0

    local_date_from = dt_from.astimezone(point_from.pytz).date()

    for segment in find_suburban(point_from, point_to, local_date_from):
        # Если рейс ходит раньше, чем нам нужно - пропускаем
        if segment.departure < dt_from:
            continue

        # Если рейс ходит позже, чем нам нужно - заканчиваем
        if segment.departure >= dt_from + timedelta(days=1):
            break

        yield segment

        count += 1

        if count >= n:
            break


def find_suburban(point_from, point_to, local_date_from):
    return find(point_from, point_to, local_date_from, 'suburban')


@jsonp_response
def directions(request):
    zone = SuburbanZone.objects.get(id=request.GET['zone'])

    response = []

    for direction in zone.externaldirection_set.order_by('title'):
        response.append({
            'id': direction.id,
            'name': direction.L_title(),
            })

    return response


def get_stations(zone_id, direction_id):
    if direction_id:
        stations = ExternalDirection.objects.get(id=direction_id).stations.order_by('externaldirectionmarker__order')
    else:
        stations = Station.objects.filter(hidden=False, externaldirectionmarker__external_direction__suburban_zone__id=zone_id).order_by('title').distinct()

        add_directions(stations)

    return stations


@jsonp_response
def stations(request):
    zone_id = request.GET['zone']

    direction_id = request.GET.get('direction')

    response = []

    for station in get_stations(zone_id, direction_id):
        d_title = getattr(station, 'direction_title', None)

        if d_title:
            name = u"%s (%s)" % (station.L_suburban_title(), d_title)
        else:
            name = station.L_suburban_title()

        response.append({
            'id': station.id,
            'name': name,
            })

    return response


@jsonp_response
def get_wizard_routes(request):
    """ временная ручка для тестирования mds-sync в qloud"""
    if request.GET.get('fail'):
        raise Exception('manual fail')

    from travel.rasp.suburban_widget.suburban_widget.views.helpers.suburbanwizard import get_wizard_routes
    routes = get_wizard_routes()

    return routes


def get_wizard_data(request):
    try:
        station_from_id, station_to_id = suburbanwizard.search_direction(request)

        station_from = Station.objects.get(id=station_from_id)
        station_to = Station.objects.get(id=station_to_id)
    except (KeyError, Station.DoesNotExist):
        return

    zone = station_from.suburban_zone

    return zone, station_from, station_to


def add_directions(stations):
    groups = dict()

    for s in stations:
        groups.setdefault(s.title, []).append(s)

    dmarkers = ExternalDirectionMarker.objects.filter(station__in=[s for g in groups.values() if len(g) > 1 for s in g])

    directions_mapping = {}

    all_directions = set()

    for dm in dmarkers:
        directions_mapping.setdefault(dm.station_id, []).append(dm.external_direction_id)
        all_directions.add(dm.external_direction_id)

    all_directions = ExternalDirection.objects.in_bulk(all_directions)

    for group in groups.values():
        if len(group) > 1:
            for i, s in enumerate(group):
                station_directions = set(directions_mapping.get(s.id, []))

                for other in chain(group[:i], group[i + 1:]):
                    station_directions.difference_update(directions_mapping.get(other.id, []))

                if station_directions:
                    station_direction = all_directions.get(station_directions.pop())

                    if station_direction:
                        s.direction_title = station_direction.L_cutted_full_title()


def next_trains_settings(request):
    zones = list(SuburbanZone.objects.order_by('settlement__title'))  # Settlement'ы закешированы

    rasp_city = request.GET.get('rasp_city')

    rasp_direction = request.GET.get('rasp_direction')

    rasp_from, rasp_to = request.GET.get('rasp_from'), request.GET.get('rasp_to')

    zone = directions = direction = stations = station_from = station_to = None

    if rasp_city and rasp_from and rasp_to:
        for z in zones:
            if unicode(z.id) == rasp_city:
                zone = z

        directions = list(zone.externaldirection_set.order_by('title'))

        for d in directions:
            if unicode(d.id) == rasp_direction:
                direction = d

        if direction:
            stations = list(direction.stations.filter(hidden=False).order_by('externaldirectionmarker__order'))
        else:
            stations = list(zone.station_set.order_by('title').filter(hidden=False))

            add_directions(stations)

        for s in stations:
            if unicode(s.id) == rasp_from:
                station_from = s

            if unicode(s.id) == rasp_to:
                station_to = s

    if not (station_from and station_to):
        # Не достались из параметров, пробуем колдунщик
        wizard_data = get_wizard_data(request)

        if wizard_data:
            zone, station_from, station_to = wizard_data

            # Общие направления для станций
            common_directions_ids = set(
                station_from.externaldirectionmarker_set.values_list('external_direction_id', flat=True)
            ).intersection(
                station_to.externaldirectionmarker_set.values_list('external_direction_id', flat=True)
            )

            direction = None

            directions = list(zone.externaldirection_set.order_by('title'))

            # Находим общее направление станций, если оно есть
            for d in directions:
                if d.id in common_directions_ids:
                    direction = d
                    break

    if not (station_from and station_to):
        # Если и в колдунщике не нашли, то выбираем согласно алгоритму RASP-3286
        # в направлениях выбираем "все направления" (придётся добавить такой пункт)
        # в обоих селектах станций выбираем первый попавшийся вокзал этого города с максимальной важностью
        zone = request.client_city.suburban_zone or Settlement.get_default_city().suburban_zone

        directions = list(zone.externaldirection_set.order_by('title'))

        stations = None

    if not stations:
        stations = get_stations(zone.id, direction and direction.id)

    context = {
        'zones': zones,
        'zone': zone,
        'directions': directions,
        'direction': direction,
        'station_from': station_from,
        'station_to': station_to,
        'stations': stations,
        'n': int_or_default(request.GET.get('n'), 5),
        'css': settings.STATIC_URL + 'next-trains-settings/_next-trains-settings.css',
        'ie_css': settings.STATIC_URL + 'next-trains-settings/_next-trains-settings.ie.css',
        'js': settings.STATIC_URL + 'next-trains-settings/next-trains-settings.js',
        }

    return render_to_response('next-trains-settings.html', context, request=request)
