package containers

import (
	"sort"

	"a.yandex-team.ru/travel/rasp/suggests/models"
)

// A PriorityQueue implements heap.Interface and holds Items.
//Если использовать только TryPushItem, то накапливает в себе limit максимальных элементов последовательности.
//При этом, на вершине кучи лежит всегда минимальный элемент из элементов кучи

type PriorityQueue struct {
	data  []models.WeightedTitleData
	limit int
	size  int
}

func (pq *PriorityQueue) Len() int {
	return pq.size
}

func (pq *PriorityQueue) Less(i, j int) bool {
	return (pq.data[i]).Less(pq.data[j])
}

func (pq *PriorityQueue) Swap(i, j int) {
	pq.data[i], pq.data[j] = pq.data[j], pq.data[i]
}

func (pq *PriorityQueue) siftUp(pos int) {
	for pos != 0 {
		parentInd := (pos - 1) / 2
		if !pq.Less(pos, parentInd) {
			return
		}
		pq.Swap(parentInd, pos)
		pos = parentInd
	}
}

func (pq *PriorityQueue) siftDown(pos int) {
	for {
		child := pos*2 + 1
		if child >= pq.size {
			return
		}
		if child+1 < pq.size && pq.Less(child+1, child) {
			child++
		}
		if pq.Less(pos, child) {
			return
		}
		pq.Swap(pos, child)
		pos = child
	}
}

// Push appends a value into heap.
func (pq *PriorityQueue) Push(x models.WeightedTitleData) {
	pq.data[pq.size] = x
	pq.size++
	pq.siftUp(pq.size - 1)
}

// Pop removes and returns an item on top of the heap.
func (pq *PriorityQueue) Pop() models.WeightedTitleData {
	res := pq.data[0]
	pq.Swap(0, pq.size-1)
	pq.size--
	if pq.size > 0 {
		pq.siftDown(0)
	}
	return res
}

// Peek returns a value on top of the heap.
func (pq *PriorityQueue) Peek() models.WeightedTitleData {
	return pq.data[0]
}

//Если элемент больше минимального элемента кучи, то он добавляется,
//При этом, если досьтигнут лимит размера, то текущий минимальный элемент кучи удаляется
func (pq *PriorityQueue) TryPushItem(item models.WeightedTitleData) bool {
	if pq.size < pq.limit || pq.Peek().Less(item) {
		pq.Push(item)

		if pq.size > pq.limit {
			pq.Pop()
		}
		return true
	}
	return false
}

func (pq *PriorityQueue) GetItems() []models.WeightedTitleData {
	result := make(models.WeightedTitleDataArray, pq.size)
	for i := 0; i < pq.size; i++ {
		result[i] = pq.data[i]
	}
	sort.Sort(sort.Reverse(result))
	return result
}

func NewPriorityQueue(limit int) PriorityQueue {
	var data []models.WeightedTitleData
	if limit > 0 {
		data = make([]models.WeightedTitleData, limit+1)
	}
	queue := PriorityQueue{data, limit, 0}
	return queue
}
