package containers

import (
	"a.yandex-team.ru/travel/rasp/suggests/models"
)

type trieNode struct {
	children       map[rune]int
	childrenValues []int
	size           int
	valueSize      int
	Value          models.TitleDataArray
	IsTerminal     bool
}

// Бор. Поддерживает операции:
// - добавление значения по ключу
// - поиск значения по ключу
// - проверка наличия ключа
type Trie struct {
	data        []trieNode
	charMapping []int
}

func (trie *Trie) traverse(text []rune, createNodes bool, index int, valueSizeDelta int) (int, error) {
	if createNodes {
		trie.data[index].size++
		trie.data[index].valueSize += valueSizeDelta
	}

	if len(text) == 0 {
		return index, nil
	}
	child, ok := trie.data[index].children[text[0]]
	if !ok {
		if createNodes {
			childNode := trieNode{IsTerminal: false}
			childNode.children = make(map[rune]int)
			childNode.childrenValues = []int{}
			trie.data = append(trie.data, childNode)
			child = len(trie.data) - 1

			trie.data[index].children[text[0]] = child
			trie.data[index].childrenValues = append(trie.data[index].childrenValues, child)
		} else {
			return -1, &notInTrieError{}
		}
	}
	return trie.traverse(text[1:], createNodes, child, valueSizeDelta)
}

func (trie *Trie) addRunes(text []rune, value models.TitleDataArray, index int) {
	resultNodeIndex, err := trie.traverse(text, true, index, len(value))
	if err != nil {
		return
	}
	trie.data[resultNodeIndex].IsTerminal = true
	trie.data[resultNodeIndex].Value = value
}

func (trie *Trie) findRunes(text []rune) (models.TitleDataArray, error) {
	resultNodeIndex, err := trie.traverse(text, false, 0, 0)
	if err != nil {
		return models.TitleDataArray{}, err
	}
	resultNode := trie.data[resultNodeIndex]
	if resultNode.IsTerminal {
		return resultNode.Value, nil
	}
	return models.TitleDataArray{}, &notInTrieError{}
}

func (trie *Trie) getAllValues(result *models.TitleDataArray, index int) {
	node := trie.data[index]
	if node.IsTerminal {
		*result = append(*result, node.Value...)
	}
	for _, value := range node.childrenValues {
		trie.getAllValues(result, value)
	}
}

func (trie *Trie) getValuesByPrefix(prefix []rune) models.TitleDataArray {
	resultNodeIndex, err := trie.traverse(prefix, false, 0, 0)
	if err != nil {
		return models.TitleDataArray{}
	}
	result := make(models.TitleDataArray, 0, trie.data[resultNodeIndex].valueSize)
	trie.getAllValues(&result, resultNodeIndex)
	return result
}

// Инициализация бора.
func (trie *Trie) Init() {
	trie.data = append(trie.data, trieNode{IsTerminal: false})
	trie.data[0].children = make(map[rune]int)
	trie.data[0].childrenValues = []int{}
}

// Добавление value по ключу key.
func (trie *Trie) Add(key string, value models.TitleDataArray) {
	trie.addRunes([]rune(key), value, 0)
}

// Значение по ключу key.
func (trie *Trie) Find(key string) models.TitleDataArray {
	res, err := trie.findRunes([]rune(key))
	if err != nil {
		return models.TitleDataArray{}
	}
	return res
}

// Получение всех значений в боре, чей ключ является префиксом строки prefix (или совпадает с ней).
func (trie *Trie) GetValuesByPrefix(prefix string) models.TitleDataArray {
	return trie.getValuesByPrefix([]rune(prefix))
}

func BuildTrie(data models.TitleNameToDataMapping) Trie {
	t := Trie{}
	t.Init()
	for key, value := range data {
		t.Add(key, value)
	}
	return t
}

func BuildPointKeyTrie(data models.ObjectDataMapping) Trie {
	t := Trie{}
	t.Init()
	for key, value := range data {
		t.Add(value.PointKey, models.TitleDataArray{models.TitleData{ID: key, IsPrefix: false}})
	}
	return t
}
