package main

import (
	"flag"
	"fmt"
	"net/http"
	"os"
	"strings"
	"time"

	"github.com/fsnotify/fsnotify"

	"a.yandex-team.ru/travel/rasp/suggests/logger"
	"a.yandex-team.ru/travel/rasp/suggests/models"
	"a.yandex-team.ru/travel/rasp/suggests/search"
	"a.yandex-team.ru/travel/rasp/suggests/utils"
	"a.yandex-team.ru/travel/rasp/suggests/web"
)

const (
	defaultPort               = 8082
	minimalTimeBetweenUpdates = 5 // In minutes
)

var (
	suggestQueryTypes = [...]string{
		"all_suggests",
		"common",
		"city",
		"by_t_type",
		"station",
		"suburban",
		"pointkey",
	}
	suggestTTypes = [...]string{"plane", "train", "bus", "sea", "river", "water"}
)

func readData(langs []string, filterDataFn *string) (*search.SearchCreator, utils.IDConverter, *models.ObjectData) {
	objectData, err := utils.ReadObjectData("objs_data/objs_data.msgpack")
	if err != nil {
		logger.Panic(err.Error())
	}
	idConvertingData, err := utils.ReadIDConverter("idconverter/idconverter.msgpack")
	if err != nil {
		logger.Panic(err.Error())
	}
	idConverter := utils.BaseIDConverter{Data: &idConvertingData}

	logger.Info("Building tries.")
	var filterData *models.FilterData = nil
	if len(*filterDataFn) > 0 {
		logger.Info("Applying PointKeyFilter")
		fd, err := utils.ReadFilterData(*filterDataFn)
		if err == nil {
			filterData = &fd
		} else {
			logger.Panic(err.Error())
		}
	}

	sc := search.CreateSearch(&objectData, &langs, nil, filterData)
	return &sc, idConverter, &objectData
}

func main() {
	logger.Init()

	flag.Usage = func() {
		fmt.Printf("Usage:\n./suggests [options]\nOptions:\n")
		flag.PrintDefaults()
	}
	var port = flag.Int("p", defaultPort, "Port to listen.")
	var languagesStr = flag.String("l", "", "Supported languages.")
	var filterDataFn = flag.String("F", os.Getenv("SUGGESTS_FILTER_DATA_FN"), "Filter data file name.")

	flag.Parse()

	languages := []string{"en", "ru", "uk"}
	if len(*languagesStr) > 0 {
		languages = strings.Split(*languagesStr, ",")
	}
	logger.Infof("Using next languages: %s", languages)

	logger.Info("Reading object data.")

	sc, idConverter, objectData := readData(languages, filterDataFn)
	wa := web.NewWebApp(sc, idConverter, objectData)

	watcher, err := fsnotify.NewWatcher()
	if err != nil {
		logger.Fatal(err.Error())
	}
	defer watcher.Close()
	lastUpdate := time.Now()
	go func() {
		for {
			select {
			case event := <-watcher.Events:
				logger.Infof("File %s was modified: %s.", event.Name, event.Op)
				if time.Since(lastUpdate).Minutes() > minimalTimeBetweenUpdates {
					logger.Infof("Rebuilding all searchers.")
					sc, idConverter, objectData := readData(languages, filterDataFn)
					wa.Update(sc, idConverter, objectData)
					lastUpdate = time.Now()
				}
			case err := <-watcher.Errors:
				logger.Infof("File watching error: %v", err)
			}
		}
	}()

	err = watcher.Add("objs_data/objs_data.msgpack")
	if err != nil {
		logger.Fatal(err.Error())
	}
	err = watcher.Add("idconverter/idconverter.msgpack")
	if err != nil {
		logger.Fatal(err.Error())
	}
	if len(*filterDataFn) > 0 {
		err = watcher.Add(*filterDataFn)
		if err != nil {
			logger.Fatal(err.Error())
		}
	}

	logger.Info("Starting web application.")

	for _, queryType := range suggestQueryTypes {
		http.HandleFunc(fmt.Sprintf("/%s", queryType), wa.SearchFactory(queryType, ""))
	}
	for _, ttype := range suggestTTypes {
		http.HandleFunc(fmt.Sprintf("/ttype/%s", ttype), wa.SearchFactory("by_t_type", ttype))
	}
	http.HandleFunc("/search", wa.NewSearch)
	http.HandleFunc("/ping", web.Ping)
	if err := http.ListenAndServe(fmt.Sprintf(":%d", *port), nil); err != nil {
		logger.Fatal(err.Error())
	}
}
