package models

import (
	"fmt"
	"math/rand"
)

type StatRouteByRegionMapping map[int]int

type StatRouteToMapping map[int]StatRouteByRegionMapping

type StatRouteFromMapping map[int]StatRouteToMapping

type StatRouteMapping map[string]StatRouteFromMapping

type PrefixToTTypesMapping map[string][]int

type LanguageToPrefixMapping map[string]PrefixToTTypesMapping

type DBInfo [][]string

//easyjson:json
type TitleData struct {
	ID       int  `json:"id"`
	IsPrefix bool `json:"is_prefix"`
}

type TitleDataArray []TitleData

type TitleNameToDataMapping map[string]TitleDataArray

type TitleTypeToNameMapping map[string]TitleNameToDataMapping

type TitleLanguageToTypeMapping map[string]TitleTypeToNameMapping

type WeightInformation map[string]map[int]int

type FullTitleMapping map[string]map[string]string

type CountryTitleMapping map[string]map[string]string

type FullObjectData struct {
	Codes              map[string]string   `json:"codes"`
	CountryID          map[string]int      `json:"country_id_national"`
	CountryTitles      CountryTitleMapping `json:"country_titles_national"`
	DisputedTerritory  bool                `json:"disputed_territory"`
	FullTitles         FullTitleMapping    `json:"full_titles"`
	IsPrefix           bool                `json:"is_prefix"`
	Lang               string              `json:"lang"`
	ID                 int                 `json:"local_id"`
	MajorityID         int                 `json:"majority_id"`
	ObjID              int                 `json:"obj_id"`
	ObjType            string              `json:"obj_type"`
	PointKey           string              `json:"point_key"`
	RegionID           int                 `json:"region_id"`
	RegionTitles       map[string]string   `json:"region_titles"`
	SearchTitles       []string            `json:"search_titles"`
	SuburbanDirections map[string]string   `json:"suburban_directions"`
	TType              int                 `json:"t_type"`
	TTypeCode          string              `json:"t_type_code"`
	TTypes             []int               `json:"t_types"`
	Titles             map[string]string   `json:"titles"`
	Weights            WeightInformation   `json:"weights"`
	ZoneID             int                 `json:"zone_id"`
	Slug               string              `json:"slug"`
	PopularTitles      map[string]string   `json:"popular_titles"`
	StationTypes       map[string]string   `json:"station_types"`
	SettlementTitles   map[string]string   `json:"settlement_titles"`
}

type ObjectDataMapping map[int]FullObjectData

type ObjectData struct {
	StatRoutes      StatRouteMapping           `json:"stat_routes"`
	StationPrefixes LanguageToPrefixMapping    `json:"station_prefixes"`
	DBInfo          DBInfo                     `json:"db_info"`
	Titles          TitleLanguageToTypeMapping `json:"titles"`
	ObjectsData     ObjectDataMapping          `json:"objects_data"`
}

type SettlementMapping map[int]int

type StationMapping map[int]int

type DBLocationInfo struct {
	ID   int    `json:"id"`
	Type string `json:"type"`
}

type LocalToDBMapping map[int]DBLocationInfo

type DBToLocalMapping struct {
	Settlement SettlementMapping `json:"settlement"`
	Station    StationMapping    `json:"station"`
}

type IDConvertingData struct {
	DBToLocal DBToLocalMapping `json:"db_to_local"`
	LocalToDB LocalToDBMapping `json:"local_to_db"`
}

type FilterData struct {
	PointKeys map[string]bool
}

func (fd *FullObjectData) GetFullTitle(lang string, nationalVersion string) string {
	if len(fd.SuburbanDirections) == 0 {
		if fd.DisputedTerritory {
			return fd.FullTitles[lang][nationalVersion]
		}
		return fd.FullTitles[lang]["default"]
	}
	direction := fd.SuburbanDirections[lang]
	if fd.DisputedTerritory {
		return fmt.Sprintf("%s (%s)", fd.FullTitles[lang][nationalVersion], direction)
	}
	return fmt.Sprintf("%s (%s)", fd.FullTitles[lang]["default"], direction)
}

func (fd *FullObjectData) GetCountryTitle(lang string, nationalVersion string) string {
	if fd.DisputedTerritory {
		return fd.CountryTitles[lang][nationalVersion]
	}
	return fd.CountryTitles[lang]["default"]
}

func (fd *FullObjectData) GetCountryID(nationalVersion string) int {
	if fd.DisputedTerritory {
		return fd.CountryID[nationalVersion]
	}
	return fd.CountryID["default"]
}

func (fd *FullObjectData) IsStation() bool {
	return fd.ObjType == "station"
}

func (fd *FullObjectData) IsSettlement() bool {
	return fd.ObjType == "settlement"
}

type MainSearchResult struct {
	Data     FullObjectData
	IsPrefix bool
}

func (msr *MainSearchResult) GetFullTitle(lang string, nationalVersion string) string {
	return msr.Data.GetFullTitle(lang, nationalVersion)
}

func (info *DBLocationInfo) RawView() (int, string) {
	if info == nil {
		return 0, ""
	}
	return info.ID, info.Type
}

type WeightedTitleData struct {
	ID       int
	IsPrefix bool
	Weights  []int
}

type WeightedTitleDataArray []WeightedTitleData

func (data WeightedTitleDataArray) Len() int {
	return len(data)
}

func (data WeightedTitleDataArray) Swap(i, j int) {
	data[i], data[j] = data[j], data[i]
}

func (data WeightedTitleDataArray) Less(i, j int) bool {
	return data[i].Less(data[j])
}

func (i WeightedTitleData) Less(j WeightedTitleData) bool {
	maxLen := max(len(i.Weights), len(j.Weights))

	for ind := 0; ind < maxLen; ind++ {

		if ind >= len(j.Weights) && i.Weights[ind] > 0 {
			return false
		}
		if ind >= len(i.Weights) && j.Weights[ind] > 0 {
			return true
		}
		if ind < len(i.Weights) && ind < len(j.Weights) && i.Weights[ind] != j.Weights[ind] {
			return i.Weights[ind] < j.Weights[ind]
		}
	}
	if i.IsPrefix != j.IsPrefix {
		return !i.IsPrefix && j.IsPrefix
	}
	return i.ID < j.ID
}

func max(a, b int) int {
	if a > b {
		return a
	}
	return b
}

//easyjson:json
type ResponseEntity struct {
	ID   int    `json:"id"`
	Type string `json:"type"`
	Name string `json:"name"`
}

// HTTP-ответ.
//easyjson:json
type Response struct {
	Result []ResponseEntity `json:"result"`
	Errors []ErrorMessage   `json:"errors"`
}

// Описание ошибки.
//easyjson:json
type ErrorMessage struct {
	Message string `json:"text"`
}

//easyjson:json
type DictStyleStruct struct {
	CountryID         int    `json:"country_id"`
	DisputedTerritory bool   `json:"disputed_territory"`
	FullTitle         string `json:"full_title"`
	FullTitleEN       string `json:"full_title_en"`
	FullTitleRU       string `json:"full_title_ru"`
	FullTitleUK       string `json:"full_title_uk"`
	ID                int    `json:"id"`
	MajorityID        int    `json:"majority"`
	ObjID             int    `json:"object_id"`
	ObjType           string `json:"type"`
	Order             int    `json:"order"`
	PointKey          string `json:"point_key"`
	RegionID          int    `json:"region_id"`
	TType             string `json:"ttype"`
	Title             string `json:"title"`
	TitleEN           string `json:"title_en"`
	TitleRU           string `json:"title_ru"`
	TitleUK           string `json:"title_uk"`
	ZoneID            int    `json:"zone_id"`
	Slug              string `json:"slug"`
	PopularTitle      string `json:"popular_title"`
	StationType       string `json:"station_type"`
	SettlementTitle   string `json:"settlement_title"`
	RegionTitle       string `json:"region_title"`
	CountryTitle      string `json:"country_title"`
}

//easyjson:json
type DictStyleResponse struct {
	Suggests      []DictStyleStruct `json:"suggests"`
	TotalFound    int               `json:"total_found"`
	HasExactMatch bool              `json:"has_exact_match"`
}

//easyjson:json
type NewStyleStruct struct {
	ObjID        int    `json:"objId"`
	ObjType      string `json:"objType"`
	FullTitle    string `json:"full_title"`
	Title        string `json:"title"`
	Slug         string `json:"slug"`
	Settlement   string `json:"settlement"`
	Region       string `json:"region"`
	Country      string `json:"country"`
	PopularTitle string `json:"popular_title"`
	StationType  string `json:"station_type"`
	PointKey     string `json:"point_key"`
}

//easyjson:json
type NewStyleResponse struct {
	Items []NewStyleStruct `json:"items"`
}

//easyjson:json
type OldStyleResponse [][4]string

//easyjson:json
type IntSlice []int

const maxWeightCount = 8
const minWeightCount = 1
const maxWeight = 1000

func GenerateWeightedTitleData(id int) WeightedTitleData {
	k := rand.Intn(maxWeightCount-minWeightCount) + minWeightCount
	buf := make([]int, k)
	for j := 0; j < k; j++ {
		buf[j] = rand.Intn(maxWeight)
	}
	return WeightedTitleData{id, true, buf}
}

func BuildWeightedTitleDataArrayCases(size int) WeightedTitleDataArray {
	res := make(WeightedTitleDataArray, 0, size)
	for i := 0; i < size; i++ {
		res = append(res, GenerateWeightedTitleData(i))
	}
	return res
}
