package search

import (
	"sync"

	"a.yandex-team.ru/travel/rasp/suggests/containers"
	"a.yandex-team.ru/travel/rasp/suggests/logger"
	"a.yandex-team.ru/travel/rasp/suggests/models"
)

const (
	compositeSearchLimit = 10
)

// Получает на вход очередь finder'ов:
// [[f1, f2],
//  [f3],
//  [f4, f5, f6]]
// f1, f2 работают параллельно, после их выполнения работает f3, потом
// параллельно f4, f5, f6.
type CompositeSearch struct {
	useCache    bool
	emptyCache  models.TitleDataArray
	cacheSaved  bool
	finderQueue [][]ChannelFinder
	cacheFinder *FullFinder
}

func NewCompositeSearch(finderQueue [][]ChannelFinder, useCache bool, cacheFinder *FullFinder) CompositeSearch {
	return CompositeSearch{useCache, models.TitleDataArray{}, false, finderQueue, cacheFinder}
}

func (cs CompositeSearch) Find(request string, limit int) (models.TitleDataArray, error) {
	if cs.cacheSaved && request == "" {
		logger.Info("Getting request from cache in composite search.")
		res := make(models.TitleDataArray, len(cs.emptyCache))
		copy(res, cs.emptyCache)
		return res, nil
	}
	if cs.useCache && request == "" {
		return (*cs.cacheFinder).Find("", limit)
	}

	u := containers.NewUniqualizer()

	for _, curFinders := range cs.finderQueue {
		resChan := make(chan models.TitleData, compositeSearchLimit)
		wg := &sync.WaitGroup{}
		wg.Add(len(curFinders))

		for _, finder := range curFinders {
			go finder.FindToChan(request, limit, resChan, wg, u.Len() > 0)
		}

		done := make(chan interface{}, 1)
		go func(chan interface{}) {
			defer close(done)
			for i := range resChan {
				u.AddItem(i)
			}
		}(done)

		wg.Wait()
		close(resChan)
		<-done

		if len(u.Items) >= limit {
			break
		}
	}
	return u.Items, nil
}

func (cs *CompositeSearch) CacheEmptyRequest() {
	res, err := cs.Find("", maxLimit)
	if err == nil {
		cs.cacheSaved = true
		cs.emptyCache = res
	}
}

func (cs CompositeSearch) FindToChan(text string, limit int, ch chan models.TitleData, wg *sync.WaitGroup, hasData bool) {
	res, _ := cs.Find(text, limit)
	for _, i := range res {
		ch <- i
	}
	wg.Done()
}
