package search

import (
	"strings"
	"sync"

	"a.yandex-team.ru/travel/rasp/suggests/containers"
	"a.yandex-team.ru/travel/rasp/suggests/models"
	"a.yandex-team.ru/travel/rasp/suggests/utils"
)

const (
	FindToChanLimit = 100
)

// Интерфейс, реализации которого используются для поиска.
type Finder interface {
	Find(text string, limit int) (models.TitleDataArray, error)
}

type ChannelFinder interface {
	FindToChan(text string, limit int, ch chan models.TitleData, wg *sync.WaitGroup, hasData bool)
}

type FullFinder interface {
	Finder
	ChannelFinder
}

type BaseFinder struct {
	trie              *containers.Trie
	filter            suggestFilter
	objectDataMapping *models.ObjectDataMapping
}

func NewBaseFinder(trie *containers.Trie, filter suggestFilter, dataMapping *models.ObjectDataMapping) BaseFinder {
	return BaseFinder{trie, filter, dataMapping}
}

func (bf BaseFinder) FindToChan(text string, limit int, ch chan models.TitleData, wg *sync.WaitGroup, hasData bool) {
	if wg != nil {
		defer wg.Done()
	}
	for _, item := range bf.trie.GetValuesByPrefix(text) {
		if bf.filter == nil || bf.filter((*bf.objectDataMapping)[item.ID]) {
			ch <- item
		}
	}
}

// Отбрасывает известные префиксы, потом ищет обычным алгоритмом.
type PrefixFinder struct {
	baseFinder      *BaseFinder
	stationPrefixes *models.PrefixToTTypesMapping
	objectsData     *models.ObjectDataMapping
}

func NewPrefixFinder(baseFinder *BaseFinder, stationPrefixes *models.PrefixToTTypesMapping, objectsData *models.ObjectDataMapping) PrefixFinder {
	return PrefixFinder{baseFinder, stationPrefixes, objectsData}
}

func (pf PrefixFinder) FindToChan(text string, limit int, ch chan models.TitleData, wg *sync.WaitGroup, hasData bool) {
	if wg != nil {
		defer wg.Done()
	}

	if text != "" {
		for prefix, ttypes := range *pf.stationPrefixes {
			if !strings.HasPrefix(text, prefix) {
				continue
			}
			rest := strings.TrimSpace(strings.TrimPrefix(text, prefix))
			if rest != "" {
				unfiltredChan := make(chan models.TitleData, FindToChanLimit)
				unfiltredWg := &sync.WaitGroup{}
				unfiltredWg.Add(1)
				go pf.baseFinder.FindToChan(rest, limit, unfiltredChan, unfiltredWg, hasData)
				done := make(chan interface{}, 1)
				go func(done chan interface{}) {
					defer close(done)
					for i := range unfiltredChan {
						if pf.filterStationByTTypes(i, ttypes) {
							ch <- i
						}
					}
				}(done)
				unfiltredWg.Wait()
				close(unfiltredChan)
				<-done
			}
		}
	}
}

func (pf PrefixFinder) filterStationByTTypes(item models.TitleData, ttypes []int) bool {
	data, ok := (*pf.objectsData)[item.ID]
	return ok && data.IsStation() && utils.IntInSlice(data.TType, ttypes)
}

type EqualFinder struct {
	trie              *containers.Trie
	filter            suggestFilter
	objectDataMapping *models.ObjectDataMapping
}

func NewEqualFinder(trie *containers.Trie, filter suggestFilter, dataMapping *models.ObjectDataMapping) EqualFinder {
	return EqualFinder{trie, filter, dataMapping}
}

func (ef EqualFinder) FindToChan(text string, limit int, ch chan models.TitleData, wg *sync.WaitGroup, hasData bool) {
	if wg != nil {
		defer wg.Done()
	}
	for _, item := range ef.trie.Find(text) {
		if ef.filter == nil || ef.filter((*ef.objectDataMapping)[item.ID]) {
			ch <- item
		}
	}
}
