package search

import (
	"fmt"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/rasp/suggests/models"
	"a.yandex-team.ru/travel/rasp/suggests/utils"
)

const (
	russiaGeoID = 225
)

type MainSearch struct {
	finder        Finder
	statThreshold int
	objectsData   *models.ObjectDataMapping
	statRoutes    *models.StatRouteMapping
	idConverter   *utils.IDConverter
}

func NewMainSearch(finder Finder, statThreshold int, objectsData *models.ObjectDataMapping, statRoutes *models.StatRouteMapping, idConverter *utils.IDConverter) MainSearch {
	return MainSearch{finder, statThreshold, objectsData, statRoutes, idConverter}
}

func (ms MainSearch) Find(text string, limit int, region int, from *models.DBLocationInfo, ttypes []string, logger log.Logger) ([]models.MainSearchResult, error) {
	data, err := ms.finder.Find(text, limit)
	if err != nil {
		return []models.MainSearchResult{}, err
	}
	var fromID int
	if from != nil {
		fromID = (*ms.idConverter).DatabaseIDToLocal(*from)
	} else {
		fromID = -1
	}
	logger.Debug(fmt.Sprintf("Searching suggests from %d, with region %d.\n", fromID, region))
	res, err := ms.sort(data, limit, region, fromID, ttypes)
	if err != nil {
		return res, err
	}
	return res, nil
}

func (ms MainSearch) sort(items models.TitleDataArray, limit int, region int, from int, ttypes []string) ([]models.MainSearchResult, error) {
	if region == -1 {
		region = russiaGeoID
	}
	regions, err := utils.GetParents(region)
	if err != nil {
		return []models.MainSearchResult{}, err
	}
	var routes models.StatRouteToMapping
	if from != -1 {
		for _, ttype := range ttypes {
			// TODO: use many ttypes (TODO из старого кода)
			if cres, ok := (*ms.statRoutes)[ttype][from]; ok {
				routes = cres
			}
		}
	}
	var routeItemsFound models.WeightedTitleDataArray
	var unroutedItems models.TitleDataArray
	if len(routes) > 0 {
		spltr := utils.WeightSplitter{
			Regions:   regions,
			Limit:     limit,
			Threshold: routesStatThreshold,
			Routes:    routes,
		}
		routedItems, restItems := spltr.SplitData(items, true, "")
		routeItemsFound, unroutedItems = routedItems, restItems
	} else {
		unroutedItems = items
	}

	if len(routeItemsFound) >= limit {
		return ms.makeResult(
			routeItemsFound,
			models.WeightedTitleDataArray{},
			models.WeightedTitleDataArray{},
			limit,
		), nil
	}

	ttype := ttypes[0]
	spltr := utils.WeightSplitter{
		Regions:     regions,
		Limit:       limit,
		Threshold:   routesStatThreshold,
		ObjectsData: ms.objectsData,
	}
	statItems, synthItems := spltr.SplitData(unroutedItems, false, ttype)
	statItemsFound := statItems

	if len(routeItemsFound)+len(statItems) >= limit {
		return ms.makeResult(
			routeItemsFound,
			statItemsFound,
			models.WeightedTitleDataArray{},
			limit,
		), nil
	}

	synthItemsFound := ms.sortSynth(synthItems, limit, regions)

	return ms.makeResult(
		routeItemsFound,
		statItemsFound,
		synthItemsFound,
		limit,
	), nil
}

func (ms MainSearch) setSynthWeight(item models.TitleData, regions []int) models.WeightedTitleData {
	result := models.WeightedTitleData{
		Weights:  make([]int, len(regions)+1),
		ID:       item.ID,
		IsPrefix: item.IsPrefix,
	}
	weights := (*ms.objectsData)[item.ID].Weights
	for index, regID := range regions {
		result.Weights[index] = weights["synth"][regID]
	}
	result.Weights[len(regions)] = weights["synth"][-1]
	return result
}

func (ms MainSearch) sortSynth(items models.TitleDataArray, limit int, regions []int) models.WeightedTitleDataArray {
	itemsWithKeys := make(models.WeightedTitleDataArray, len(items))
	for index, item := range items {
		itemsWithKeys[index] = ms.setSynthWeight(item, regions)
	}
	return utils.Nlargest(limit, itemsWithKeys)
}

func (ms MainSearch) addItems(storage *[]models.MainSearchResult, items models.WeightedTitleDataArray, limit int) bool {
	for i := range items {
		*storage = append(*storage, models.MainSearchResult{
			Data:     (*ms.objectsData)[items[i].ID],
			IsPrefix: items[i].IsPrefix,
		})
		if len(*storage) == limit {
			return false
		}
	}
	return true
}

func (ms MainSearch) makeResult(routeItems, statItems, synthItems models.WeightedTitleDataArray, limit int) []models.MainSearchResult {
	result := make([]models.MainSearchResult, 0, limit)
	needMore := ms.addItems(&result, routeItems, limit)
	if needMore {
		needMore = ms.addItems(&result, statItems, limit)
	}
	if needMore {
		ms.addItems(&result, synthItems, limit)
	}

	return result
}
