package search

import (
	"a.yandex-team.ru/travel/rasp/suggests/logger"
	"a.yandex-team.ru/travel/rasp/suggests/speller"
	"a.yandex-team.ru/travel/rasp/suggests/utils"
)

const (
	spellerResultLimit = 2
)

// Интерфейс, реализации которого преобразуют запрос к набору строк, по которым
// будет производиться поиск.
type RequestPreparer interface {
	GetTexts(request string) []string
}

type SelfsameRequestPreparer struct{}

func (requestPreparer SelfsameRequestPreparer) GetTexts(request string) []string {
	result := utils.PrepareTitleText(request)
	logger.Debugf("Using %#v unchanged", result)
	return []string{result}
}

type TransiltRequestPreparer struct {
	language string
}

func NewTransiltRequestPreparer(lang string) TransiltRequestPreparer {
	return TransiltRequestPreparer{lang}
}

func (rp TransiltRequestPreparer) GetTexts(request string) []string {
	prepared := utils.PrepareTitleText(request)
	var tableName string
	if rp.language == "ru" {
		tableName = "lat-cyr"
	} else {
		tableName = "cyr-lat"
	}
	translited := utils.Transliterate(prepared, tableName)
	if translited != "" && translited != request {
		logger.Debugf("Translited %#v to %#v", request, translited)
		return []string{translited}
	}
	return []string{}
}

type KeyboardLayoutRequestPreparer struct {
	language string
}

func NewKeyboardLayoutRequestPreparer(lang string) KeyboardLayoutRequestPreparer {
	return KeyboardLayoutRequestPreparer{lang}
}

func (rp KeyboardLayoutRequestPreparer) GetTexts(request string) []string {
	variants := utils.GetLayoutVariants(request, rp.language)
	logger.Debugf("Changed keyboard layout from %s to %#v.", request, variants)
	for i := range variants {
		variants[i] = utils.PrepareTitleText(variants[i])
	}
	return variants
}

type SpellerRequestPreparer struct {
	language string
	speller  *speller.Speller
}

func NewSpellerRequestPreparer(lang string, speller speller.Speller) SpellerRequestPreparer {
	return SpellerRequestPreparer{lang, &speller}
}

func (rp SpellerRequestPreparer) GetTexts(request string) []string {
	req := utils.PrepareTitleText(request)
	bestLocs := (*rp.speller).GetMostSimilarLocations(req, spellerResultLimit, rp.language)
	res := make([]string, len(bestLocs))
	for i := range bestLocs {
		res[i] = utils.PrepareTitleText(bestLocs[i].Location.Titles[rp.language])
	}
	logger.Debugf("Changed %s to %#v with speller.", request, res)
	return res
}
