package search

import (
	"a.yandex-team.ru/travel/rasp/suggests/containers"
	"a.yandex-team.ru/travel/rasp/suggests/logger"
	"a.yandex-team.ru/travel/rasp/suggests/models"
	"a.yandex-team.ru/travel/rasp/suggests/speller"
	"a.yandex-team.ru/travel/rasp/suggests/utils"
)

const (
	routesStatThreshold = 10
)

var (
	defaultTitles = []string{"ru", "en", "uk"}
)

func createTTypeSearch(lang string, ttype string, trie *containers.Trie, stationPrefixes *models.PrefixToTTypesMapping,
	objectsData *models.ObjectDataMapping, filter suggestFilter, locations *[]models.FullObjectData,
	locationPrecalc speller.PrecalculatedData) CompositeSearch {
	bf := NewBaseFinder(trie, filter, objectsData)
	pf := NewPrefixFinder(&bf, stationPrefixes, objectsData)
	//splr := speller.NewLevenshteinSpeller(locations, locationPrecalc, objectsData, false)
	return NewCompositeSearch([][]ChannelFinder{
		{
			NewSearchEngine(
				bf,
				SelfsameRequestPreparer{},
				"Basic search",
				false,
			),
			NewSearchEngine(
				pf,
				SelfsameRequestPreparer{},
				"Prefix search",
				false,
			),
		},
		{
			NewSearchEngine(
				bf,
				NewKeyboardLayoutRequestPreparer(lang),
				"Basic keyboard-layout search",
				false,
			),
			NewSearchEngine(
				pf,
				NewKeyboardLayoutRequestPreparer(lang),
				"Prefix keyboard-layout search",
				false,
			),
		},
		{
			NewSearchEngine(
				bf,
				NewTransiltRequestPreparer(lang),
				"Basic translit search",
				false,
			),
			NewSearchEngine(
				pf,
				NewTransiltRequestPreparer(lang),
				"Prefix translit search",
				false,
			),
		},
		//{
		//    NewSearchEngine(
		//        bf,
		//        NewSpellerRequestPreparer(lang, &splr),
		//        "Speller search",
		//        true,
		//    ),
		//},
	}, false, nil)
}

func createPointKeySearch(trie *containers.Trie, objectsData *models.ObjectDataMapping) CompositeSearch {
	ef := NewEqualFinder(trie, nil, objectsData)
	return NewCompositeSearch([][]ChannelFinder{
		{
			NewSearchEngine(
				ef,
				SelfsameRequestPreparer{},
				"Basic search",
				false,
			),
		},
	}, false, nil)
}

func CreateSearch(data *models.ObjectData, langs *[]string, ttypes *[]string, filterData *models.FilterData) SearchCreator {
	locations := make([]models.FullObjectData, 0, len(data.ObjectsData))
	for _, loc := range data.ObjectsData {
		locations = append(locations, loc)
	}
	locationPrecalc := make(speller.PrecalculatedData)
	if langs == nil {
		langs = &defaultTitles
	}
	searchesTypes := map[string]suggestFilter{
		"searches_by_lang":            nil,
		"settlement_searches_by_lang": filterSettlementSuggests,
		"station_searches_by_lang":    filterStationSuggests,
	}
	if filterData != nil {
		filterByFilterData := filterByFilterDataFactory(filterData)
		for t, f := range searchesTypes {
			searchesTypes[t] = nestedFactory(f, filterByFilterData)
		}
	}

	searches := make(finderMapping)
	pointKeyTrie := containers.BuildPointKeyTrie(data.ObjectsData)
	for lang, ttypesData := range data.Titles {
		if langs != nil && !utils.StringInSlice(lang, *langs) {
			continue
		}
		locationPrecalc.PrecalculateLocations(lang, locations)

		stationPrefixes := data.StationPrefixes[lang]
		for ttype, titleData := range ttypesData {
			if ttypes != nil && !utils.StringInSlice(ttype, *ttypes) {
				continue
			}
			logger.Debugf("Building a new trie: %s, %s.", lang, ttype)
			trie := containers.BuildTrie(titleData)

			for searchType, filterFunc := range searchesTypes {
				search := createTTypeSearch(lang, ttype, &trie, &stationPrefixes, &data.ObjectsData, filterFunc, &locations, locationPrecalc)
				search.CacheEmptyRequest()
				searches.SetVal(searchType, lang, ttype, search)
			}
		}
		search := createPointKeySearch(&pointKeyTrie, &data.ObjectsData)
		search.CacheEmptyRequest()
		searches.SetVal("pointkey", lang, "all", search)
	}

	return NewSearchCreator(searches, &data.ObjectsData, &data.StatRoutes)
}
