package search

import (
	"strings"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/rasp/suggests/models"
	"a.yandex-team.ru/travel/rasp/suggests/utils"
)

const (
	maxLimit = 25
)

type finderMapping map[string]map[string]map[string]FullFinder

func (fm *finderMapping) SetVal(searchType, lang, ttype string, finder FullFinder) {
	fmp := *fm
	if _, ok := fmp[searchType]; !ok {
		fmp[searchType] = make(map[string]map[string]FullFinder)
	}
	if _, ok := fmp[searchType][lang]; !ok {
		fmp[searchType][lang] = make(map[string]FullFinder)
	}
	fmp[searchType][lang][ttype] = finder
}

type SearchCreator struct {
	finders          finderMapping
	objectsData      *models.ObjectDataMapping
	statRoutes       *models.StatRouteMapping
	emptySearchCache *EmptySearchCache
}

func NewSearchCreator(finders finderMapping, objectsData *models.ObjectDataMapping, statRoutes *models.StatRouteMapping) SearchCreator {
	cache := NewEmptySearchCache()
	return SearchCreator{finders, objectsData, statRoutes, &cache}
}

func (sc *SearchCreator) Find(text string, limit int, region int, from *models.DBLocationInfo, ttypes []string, lang string,
	nationalVersion string, suggestType string, idConverter utils.IDConverter, logger log.Logger) ([]models.MainSearchResult, error) {
	if limit > maxLimit {
		limit = maxLimit
	}
	emptySearch := strings.TrimSpace(text) == ""
	if emptySearch {
		if res, ok := sc.emptySearchCache.GetValue(limit, region, from, ttypes, lang, nationalVersion, suggestType); ok {
			logger.Debug("Getting response for empty request from cache.")
			return res, nil
		}
	}

	searchType := "searches_by_lang"
	if suggestType == "settlement" {
		searchType = "settlement_searches_by_lang"
	} else if suggestType == "station" {
		searchType = "station_searches_by_lang"
	} else if suggestType == "pointkey" {
		searchType = "pointkey"
	}
	findersByST := sc.finders[searchType]
	langs := utils.GetLangsByNationalVersion(nationalVersion, lang)
	finders := make([][]ChannelFinder, 0, len(langs)*len(ttypes))
	var cacheFinder *FullFinder
	for _, l := range langs {
		findersByLang, ok := findersByST[l]
		if !ok {
			continue
		}
		for _, ttype := range ttypes {
			if finder, ok := findersByLang[ttype]; ok {
				if cacheFinder == nil {
					cacheFinder = &finder
				}
				finders = append(finders, []ChannelFinder{finder})
			}
		}
	}
	cs := NewCompositeSearch(finders, true, cacheFinder)
	ms := NewMainSearch(cs, routesStatThreshold, sc.objectsData, sc.statRoutes, &idConverter)
	if emptySearch {
		res, err := ms.Find(text, maxLimit, region, from, ttypes, logger)
		if err == nil {
			logger.Debug("Setting response for empty request to cache.")
			sc.emptySearchCache.SetValue(res, region, from, ttypes, lang, nationalVersion, suggestType)
		} else {
			sc.emptySearchCache.SetValue([]models.MainSearchResult{}, region, from, ttypes, lang, nationalVersion, suggestType)
		}
		return res[:limit], err
	} else {
		return ms.Find(text, limit, region, from, ttypes, logger)
	}
}
