package speller

import (
	"sort"
	"strings"
	"unicode/utf8"

	"a.yandex-team.ru/travel/rasp/suggests/logger"
	"a.yandex-team.ru/travel/rasp/suggests/models"
	"a.yandex-team.ru/travel/rasp/suggests/utils"
)

const (
	// maxLengthDifference is such value that if the difference between lengths is bigger, the word is skipped.
	maxLengthDifference = 2
	// infinity is a local "infinity" number, must be greater than anything else.
	infinity          = 1000000
	distanceToPrecalc = 2
	maxPrefixLength   = 7
)

type SpellerResponseEntity struct {
	Location models.FullObjectData
	Distance int
	Shorten  bool
}

type SpellerResponseEntities []SpellerResponseEntity

func (data SpellerResponseEntities) Len() int {
	return len(data)
}

func (data SpellerResponseEntities) Less(i, j int) bool {
	return data[i].Distance < data[j].Distance
}

func (data SpellerResponseEntities) Swap(i, j int) {
	data[i], data[j] = data[j], data[i]
}

type Speller interface {
	GetMostSimilarLocations(string, int, string) SpellerResponseEntities
}

type LevenshteinSpeller struct {
	locations       *[]models.FullObjectData
	locationPrecalc PrecalculatedData
	objectsData     *models.ObjectDataMapping
	searchByPrefix  bool
}

type PrecalculatedData map[string]map[string][]PrecalcedItem

func (locationPrecalc PrecalculatedData) PrecalculateLocations(lang string, locations []models.FullObjectData) {
	locationPrecalc[lang] = make(map[string][]PrecalcedItem)
	for i, loc := range locations {
		if i%10000 == 0 {
			logger.Debugf("Precalculation: %d / %d", i, len(locations))
		}
		title := utils.Substr(strings.ToLower(loc.Titles[lang]), 0, maxPrefixLength)
		locationPrecalc[lang][title] = append(locationPrecalc[lang][title], PrecalcedItem{ID: loc.ID, Difference: 0})
		for i := 1; i <= distanceToPrecalc; i++ {
			for _, c := range RemoveNChars(title, i) {
				locationPrecalc[lang][c] = append(locationPrecalc[lang][c], PrecalcedItem{ID: loc.ID, Difference: i})
			}
		}
	}
}

type PrecalcedItem struct {
	ID         int
	Difference int
}

func NewLevenshteinSpeller(locations *[]models.FullObjectData, locationPrecalc PrecalculatedData, objectsData *models.ObjectDataMapping, searchByPrefix bool) LevenshteinSpeller {
	return LevenshteinSpeller{locations, locationPrecalc, objectsData, searchByPrefix}
}

func addPrecalculatedLocation(location PrecalcedItem, dist int, dists map[int]int, reqIDs *[]int) {
	oldRes, ok := dists[location.ID]
	newRes := location.Difference + dist
	if !ok {
		*reqIDs = append(*reqIDs, location.ID)
		dists[location.ID] = newRes
	} else if oldRes > newRes {
		dists[location.ID] = newRes
	}
}

func (ls *LevenshteinSpeller) getSpellerResponseEntity(distPrediction int, locName string, request string, reqLen int, worstDistance int,
	location models.FullObjectData, buf []int) (SpellerResponseEntity, bool) {
	if distPrediction > distanceToPrecalc {
		locLen := utf8.RuneCountInString(locName)
		if abs(reqLen-locLen) <= min(maxLengthDifference, worstDistance) {
			return handleLevenshteinDistance(request, locName, location, false, buf, worstDistance), true
		} else if ls.searchByPrefix && reqLen < locLen {
			return handleLevenshteinDistance(request, string([]rune(locName)[:reqLen]), location, true, buf, worstDistance), true
		}
	} else {
		return SpellerResponseEntity{location, distPrediction, false}, true
	}
	return SpellerResponseEntity{}, false
}

func (ls *LevenshteinSpeller) GetMostSimilarLocations(request string, limit int, lang string) SpellerResponseEntities {
	reqLen := utf8.RuneCountInString(request)
	request = strings.ToLower(request)
	reqPart := utils.Substr(request, 0, maxPrefixLength)
	worstDistance := 5

	buf := make([]int, utf8.RuneCountInString(request)+1)
	dists := make(map[int]int)
	queue := NewPriorityQueue()
	reqIDs := make([]int, 0, 200)

	for i := 1; i <= distanceToPrecalc; i++ {
		for _, reqComb := range RemoveNChars(reqPart, i) {
			for _, location := range ls.locationPrecalc[lang][reqComb] {
				addPrecalculatedLocation(location, i, dists, &reqIDs)
			}
		}
	}
	for _, location := range ls.locationPrecalc[lang][reqPart] {
		addPrecalculatedLocation(location, 0, dists, &reqIDs)
	}

	for _, reqID := range reqIDs {
		location := (*ls.objectsData)[reqID]
		locName := location.Titles[lang]
		distPrediction := dists[reqID]
		res, ok := ls.getSpellerResponseEntity(distPrediction, locName, request, reqLen, worstDistance, location, buf)

		if ok && res.Distance <= worstDistance {
			newItem := &PriorityQueueItem{
				value:    res,
				priority: res.Distance,
				shorten:  res.Shorten,
			}
			if queue.Len() < limit || queue.Peek().Less(*newItem) {
				item := &PriorityQueueItem{
					value:    res,
					priority: res.Distance,
					shorten:  res.Shorten,
				}
				queue.PushItem(item)

				if queue.Len() > limit {
					worstDistance = queue.Peek().value.Distance
					queue.PopItem()
				}
			}
		}
	}

	result := make(SpellerResponseEntities, len(queue))
	for i := range queue {
		result[i] = queue[i].value
	}
	sort.Sort(result)
	return result
}

func handleLevenshteinDistance(request, locName string, location models.FullObjectData, shorten bool, buf []int, worstDistance int) SpellerResponseEntity {
	knownTitle := strings.ToLower(locName)
	dist := levenshteinDistance(knownTitle, request, worstDistance, buf)
	return SpellerResponseEntity{location, dist, shorten}
}

func levenshteinDistance(firstStr string, secondStr string, limit int, dist []int) int {
	if limit != -1 && abs(utf8.RuneCountInString(firstStr)-utf8.RuneCountInString(secondStr)) > limit {
		return infinity
	}
	s2Len := utf8.RuneCountInString(secondStr)
	for i := 0; i <= s2Len; i++ {
		dist[i] = i
	}
	for _, firstChar := range firstStr {
		lastVal := dist[0]
		dist[0]++
		index := 0
		bestDist := dist[0]
		for _, secondChar := range secondStr {
			bestVal := min(dist[index+1], dist[index]) + 1
			if secondChar != firstChar {
				bestVal = min(bestVal, lastVal+1)
			} else {
				bestVal = min(bestVal, lastVal)
			}
			lastVal, dist[index+1] = dist[index+1], bestVal
			index++
			bestDist = min(bestDist, bestVal)
		}
		if limit != -1 && bestDist > limit {
			return infinity
		}
	}
	return dist[s2Len]
}

func min(a, b int) int {
	if a < b {
		return a
	}
	return b
}

func abs(a int) int {
	if a < 0 {
		return -a
	}
	return a
}

func Remove1Char(word string) []string {
	result := make([]string, 0, len(word))
	rw := []rune(word)
	for i := range rw {
		result = append(result, string(rw[:i])+string(rw[i+1:]))
	}
	return result
}

func factorial(n int) int {
	res := 1
	for i := 2; i <= n; i++ {
		res *= i
	}
	return res
}

func getWordSubseqNumber(word string, n int) int {
	return factorial(len(word)) / factorial(n) / factorial(len(word)-n)
}

func removeNChars(word string, n int, result *[]string) {
	if n == 1 {
		*result = append(*result, Remove1Char(word)...)
		return
	}

	for _, i := range Remove1Char(word) {
		removeNChars(i, n-1, result)
	}
}

func RemoveNChars(word string, n int) []string {
	result := make([]string, 0, getWordSubseqNumber(word, n))
	removeNChars(word, n, &result)
	return result
}
