package speller

import (
	"container/heap"
)

// A PriorityQueueItem is something we manage inside a priority queue.
type PriorityQueueItem struct {
	value    SpellerResponseEntity
	priority int
	index    int
	shorten  bool
}

func (pq PriorityQueueItem) Less(other PriorityQueueItem) bool {
	if pq.priority != other.priority {
		return pq.priority > other.priority
	}
	if pq.value.Location.Weights["all"][-1] != other.value.Location.Weights["all"][-1] {
		return pq.value.Location.Weights["all"][-1] < other.value.Location.Weights["all"][-1]
	}
	return pq.value.Location.ObjID > other.value.Location.ObjID
}

// A PriorityQueue implements heap.Interface and holds Items.
type PriorityQueue []*PriorityQueueItem

func (pq PriorityQueue) Len() int { return len(pq) }

func (pq PriorityQueue) Less(i, j int) bool {
	return (*pq[i]).Less(*pq[j])
}

func (pq PriorityQueue) Swap(i, j int) {
	pq[i], pq[j] = pq[j], pq[i]
	pq[i].index = i
	pq[j].index = j
}

// Push appends a value into heap.
func (pq *PriorityQueue) Push(x interface{}) {
	n := len(*pq)
	item := x.(*PriorityQueueItem)
	item.index = n
	*pq = append(*pq, item)
}

// Pop removes and returns an item on top of the heap.
func (pq *PriorityQueue) Pop() interface{} {
	old := *pq
	n := len(old)
	item := old[n-1]
	item.index = -1
	*pq = old[0 : n-1]
	return item
}

func (pq *PriorityQueue) update(item *PriorityQueueItem, value SpellerResponseEntity, priority int) {
	item.value = value
	item.priority = priority
	heap.Fix(pq, item.index)
}

// Peek returns a value on top of the heap.
func (pq *PriorityQueue) Peek() *PriorityQueueItem {
	return (*pq)[0]
}

func (pq *PriorityQueue) PushItem(item *PriorityQueueItem) {
	heap.Push(pq, item)
	pq.update(item, item.value, item.priority)
}

func (pq *PriorityQueue) PopItem() {
	heap.Pop(pq)
}

func NewPriorityQueue() PriorityQueue {
	var queue PriorityQueue
	heap.Init(&queue)
	return queue
}
