package utils

import (
	"bytes"
	"regexp"
	"strings"

	"a.yandex-team.ru/travel/rasp/suggests/containers"
	"a.yandex-team.ru/travel/rasp/suggests/models"
)

func UnpackTitleDataArray(data []models.TitleDataArray) models.TitleDataArray {
	resultLen := 0
	resultSize := 0
	for _, arr := range data {
		resultLen += len(arr)
	}

	result := make(models.TitleDataArray, resultLen)
	for _, arr := range data {
		for _, item := range arr {
			result[resultSize] = item
			resultSize++
		}
	}
	return result
}

var translitTables = map[string]map[rune]string{
	"cyr-lat": {
		1072: "a",
		1073: "b",
		1074: "v",
		1075: "g",
		1076: "d",
		1077: "e",
		1078: "zh",
		1079: "z",
		1080: "i",
		1081: "j",
		1082: "k",
		1083: "l",
		1084: "m",
		1085: "n",
		1086: "o",
		1087: "p",
		1088: "r",
		1089: "s",
		1090: "t",
		1091: "u",
		1092: "f",
		1093: "x",
		1094: "cz",
		1095: "ch",
		1096: "sh",
		1097: "shh",
		1098: "qd",
		1099: "qi",
		1100: "qt",
		1101: "ye",
		1102: "yu",
		1103: "ya",
		1105: "yo",
	},
	"lat-cyr": {
		97:  "а",
		98:  "б",
		99:  "ц",
		100: "д",
		101: "е",
		102: "ф",
		103: "г",
		104: "х",
		105: "и",
		106: "й",
		107: "к",
		108: "л",
		109: "м",
		110: "н",
		111: "о",
		112: "п",
		113: "я",
		114: "р",
		115: "с",
		116: "т",
		117: "у",
		118: "в",
		119: "в",
		120: "х",
		121: "ы",
		122: "з",
	},
}

func Transliterate(request string, tableName string) string {
	var buffer bytes.Buffer
	buffer.Grow(len(request))
	for _, curChar := range request {
		if newChar, ok := translitTables[tableName][curChar]; ok {
			buffer.WriteString(newChar)
		} else {
			buffer.WriteRune(curChar)
		}
	}
	return buffer.String()
}

var separators = map[rune]bool{
	// [ord(i) for i in string.whitespace]
	32: true,
	9:  true,
	10: true,
	13: true,
	11: true,
	12: true,
	// [ord(i) for i in string.punctuation]
	33:  true,
	34:  true,
	35:  true,
	36:  true,
	37:  true,
	38:  true,
	39:  true,
	40:  true,
	41:  true,
	42:  true,
	43:  true,
	44:  true,
	45:  true,
	46:  true,
	47:  true,
	58:  true,
	59:  true,
	60:  true,
	61:  true,
	62:  true,
	63:  true,
	64:  true,
	91:  true,
	92:  true,
	93:  true,
	94:  true,
	95:  true,
	96:  true,
	123: true,
	124: true,
	125: true,
	126: true,
	// nbsp
	160: true,
	// « »
	171: true,
	187: true,
}

func PrepareTitleText(request string) string {
	request = strings.ToLower(request)
	var buffer bytes.Buffer
	buffer.Grow(len(request))
	lastSep := false
	for _, c := range request {
		_, isSeparator := separators[c]
		if isSeparator {
			if !lastSep {
				buffer.WriteRune(32)
			}
			lastSep = true
		} else {
			buffer.WriteRune(c)
			lastSep = false
		}
	}

	return strings.TrimSpace(buffer.String())
}

type PuntoDirection struct {
	src string
	dst string
}

// Маппинг, полученный из:
// en: qwertyuiop{[}]asdfghjkl:;'"zxcvbnm<,>.`
// ru: йцукенгшщзххъъфывапролджжээячсмитьббююё
// uk: йцукенгшщзххїїфівапролджжєєячсмитьббююё
var puntoMapping = map[PuntoDirection]map[rune]rune{
	{"en", "ru"}: {
		34:  1101,
		39:  1101,
		44:  1073,
		46:  1102,
		58:  1078,
		59:  1078,
		60:  1073,
		62:  1102,
		91:  1093,
		93:  1098,
		96:  1105,
		97:  1092,
		98:  1080,
		99:  1089,
		100: 1074,
		101: 1091,
		102: 1072,
		103: 1087,
		104: 1088,
		105: 1096,
		106: 1086,
		107: 1083,
		108: 1076,
		109: 1100,
		110: 1090,
		111: 1097,
		112: 1079,
		113: 1081,
		114: 1082,
		115: 1099,
		116: 1077,
		117: 1075,
		118: 1084,
		119: 1094,
		120: 1095,
		121: 1085,
		122: 1103,
		123: 1093,
		125: 1098,
	}, {"ru", "en"}: {
		1072: 102,
		1073: 44,
		1074: 100,
		1075: 117,
		1076: 108,
		1077: 116,
		1078: 59,
		1079: 112,
		1080: 98,
		1081: 113,
		1082: 114,
		1083: 107,
		1084: 118,
		1085: 121,
		1086: 106,
		1087: 103,
		1088: 104,
		1089: 99,
		1090: 110,
		1091: 101,
		1092: 97,
		1093: 91,
		1094: 119,
		1095: 120,
		1096: 105,
		1097: 111,
		1098: 93,
		1099: 115,
		1100: 109,
		1101: 34,
		1102: 46,
		1103: 122,
		1105: 96,
	}, {"uk", "ru"}: {
		1072: 1072,
		1073: 1073,
		1074: 1074,
		1075: 1075,
		1076: 1076,
		1077: 1077,
		1078: 1078,
		1079: 1079,
		1080: 1080,
		1081: 1081,
		1082: 1082,
		1083: 1083,
		1084: 1084,
		1085: 1085,
		1086: 1086,
		1087: 1087,
		1088: 1088,
		1089: 1089,
		1090: 1090,
		1091: 1091,
		1092: 1092,
		1093: 1093,
		1094: 1094,
		1095: 1095,
		1096: 1096,
		1097: 1097,
		1100: 1100,
		1102: 1102,
		1103: 1103,
		1105: 1105,
		1108: 1101,
		1110: 1099,
		1111: 1098,
	}, {"ru", "uk"}: {
		1072: 1072,
		1073: 1073,
		1074: 1074,
		1075: 1075,
		1076: 1076,
		1077: 1077,
		1078: 1078,
		1079: 1079,
		1080: 1080,
		1081: 1081,
		1082: 1082,
		1083: 1083,
		1084: 1084,
		1085: 1085,
		1086: 1086,
		1087: 1087,
		1088: 1088,
		1089: 1089,
		1090: 1090,
		1091: 1091,
		1092: 1092,
		1093: 1093,
		1094: 1094,
		1095: 1095,
		1096: 1096,
		1097: 1097,
		1098: 1111,
		1099: 1110,
		1100: 1100,
		1101: 1108,
		1102: 1102,
		1103: 1103,
		1105: 1105,
	},
}

func StringInSlice(a string, list []string) bool {
	for _, b := range list {
		if b == a {
			return true
		}
	}
	return false
}

func IntInSlice(a int, list []int) bool {
	for _, b := range list {
		if b == a {
			return true
		}
	}
	return false
}

func ConvertToLayout(request string, mapping map[rune]rune) string {
	var buffer bytes.Buffer
	buffer.Grow(len(request))
	for _, c := range request {
		val, ok := mapping[c]
		if ok {
			buffer.WriteRune(val)
		} else {
			buffer.WriteRune(c)
		}
	}
	return buffer.String()
}

func GetLayoutVariants(request string, lang string) []string {
	result := make([]string, 0, len(puntoMapping))
	request = strings.ToLower(request)

	for direction, mapping := range puntoMapping {
		if direction.dst == lang {
			cres := ConvertToLayout(request, mapping)
			if cres != request && !StringInSlice(cres, result) {
				result = append(result, cres)
			}
		}
	}

	return result
}

type WeightSplitter struct {
	Regions     []int
	Limit       int
	Threshold   int
	Routes      models.StatRouteToMapping
	ObjectsData *models.ObjectDataMapping
}

type regionGetFunc func(models.FullObjectData) (models.StatRouteByRegionMapping, bool)

func (ws *WeightSplitter) SplitData(items models.TitleDataArray, useRoutes bool, ttype string) (models.WeightedTitleDataArray, models.TitleDataArray) {
	weightedItems := containers.NewPriorityQueue(ws.Limit)
	unweightedSize := 0
	topSize := len(ws.Regions)

	itemCountByWeight := make([]int, topSize)
	hasWeight := true
	var weights []int
	for _, item := range items {
		if hasWeight {
			weights = make([]int, topSize+1)
		} else if len(weights) > topSize+1 {
			weights = weights[:topSize+1]
		}
		hasWeight = false

		var weightsByRegions models.StatRouteByRegionMapping
		var ok bool

		if useRoutes {
			weightsByRegions, ok = ws.Routes[item.ID]
		} else {
			if fullData, ok1 := (*ws.ObjectsData)[item.ID]; ok1 {
				weightsByRegions, ok = fullData.Weights[ttype]
			} else {
				continue
			}
		}

		if ok {
			for regIndex, regID := range ws.Regions {
				if regIndex > topSize {
					break
				}
				weight, ok := weightsByRegions[regID]

				if !ok || weight < ws.Threshold {
					weight = 0
				}
				if weight > 0 {
					hasWeight = true
					weights[regIndex] = weight
					itemCountByWeight[regIndex]++
					if itemCountByWeight[regIndex] >= ws.Limit {
						topSize = min(topSize, regIndex)
					}
					break
				}
			}
		}

		if hasWeight {
			hasWeight = weightedItems.TryPushItem(models.WeightedTitleData{ID: item.ID, IsPrefix: item.IsPrefix, Weights: weights})
		} else {
			items[unweightedSize] = item
			unweightedSize++
		}
	}
	return weightedItems.GetItems(), items[:unweightedSize]
}

func siftDown(data models.WeightedTitleDataArray, start int, end int) {
	root := start
	for (2*root + 1) <= end {
		child := 2*root + 1
		if child+1 <= end && data[child].Less(data[child+1]) {
			child = child + 1
		}
		if data[root].Less(data[child]) {
			data[root], data[child] = data[child], data[root]
			root = child
		} else {
			return
		}
	}
}

func heapify(data models.WeightedTitleDataArray) {
	end := len(data) - 1
	start := (end - 1) / 2
	for start >= 0 {
		siftDown(data, start, end)
		start--
	}
}

func Nlargest(limit int, data models.WeightedTitleDataArray) models.WeightedTitleDataArray {
	heapify(data)
	hs := len(data)
	res := make(models.WeightedTitleDataArray, 0, limit)
	for i := 0; i < min(limit, len(data)); i++ {
		res = append(res, data[0])
		hs -= 1
		data[0], data[hs] = data[hs], data[0]
		siftDown(data, 0, hs-1)
	}
	return res
}

var langsByNationalVersion = map[string][]string{
	"ru": {"ru", "en", "uk"},
	"uk": {"uk", "ru", "en"},
}

func GetLangsByNationalVersion(nationalVersion string, defaultLang string) []string {
	result := make([]string, 0, len(langsByNationalVersion[nationalVersion])+1)
	result = append(result, defaultLang)
	for _, lang := range langsByNationalVersion[nationalVersion] {
		if lang != defaultLang {
			result = append(result, lang)
		}
	}
	return result
}

var allowedPJSONChars, _ = regexp.Compile(`^\w{1,64}$`)
var validOriginRegexp, _ = regexp.Compile(`(^https?://(.*\.)*yandex\.(ru|by|ua|kz|uz|com|com\.tr|az)$)|(^https?://(.*\.)*ya\.(ru|by|ua|kz|uz|com|com\.tr|az)$)`)

func IsValidPJSON(req string) bool {
	return allowedPJSONChars.MatchString(req)
}

func IsValidOrigin(req string) bool {
	return validOriginRegexp.MatchString(req)
}

func min(a, b int) int {
	if a < b {
		return a
	}
	return b
}

func max(a, b int) int {
	if a > b {
		return a
	}
	return b
}

func Substr(s string, start int, finish int) string {
	return string([]rune(s)[start:finish])
}
