package web

import (
	"a.yandex-team.ru/travel/rasp/suggests/logger"
	"a.yandex-team.ru/travel/rasp/suggests/models"
)

type responseFormatter func(items []models.MainSearchResult, lang, nationalVersion string) string

func oldStyleFormat(items []models.MainSearchResult, lang, nationalVersion string) string {
	respItems := make(models.OldStyleResponse, len(items))
	for i, item := range items {
		respItems[i][0] = item.Data.PointKey
		respItems[i][1] = item.Data.Titles[lang]
		respItems[i][2] = item.GetFullTitle(lang, nationalVersion)
		respItems[i][3] = item.Data.Slug
	}

	jsonItems, err := respItems.MarshalJSON()
	if err != nil {
		logger.Errorf("Error while marshalling error: %+v", err)
		return "[null, []]"
	}
	return "[null, " + string(jsonItems) + "]"
}

func allStyleFormat(items []models.MainSearchResult, lang, nationalVersion string) string {
	result := make([]models.DictStyleStruct, len(items))
	hasNotPrefix := false

	for i, item := range items {
		var id int
		if item.Data.ObjType == "country" {
			id = item.Data.ObjID
		} else {
			id = item.Data.GetCountryID(nationalVersion)
		}
		result[i] = models.DictStyleStruct{
			CountryID:         id,
			DisputedTerritory: item.Data.DisputedTerritory,
			FullTitle:         item.GetFullTitle(lang, nationalVersion),
			FullTitleEN:       item.GetFullTitle("en", nationalVersion),
			FullTitleRU:       item.GetFullTitle("ru", nationalVersion),
			FullTitleUK:       item.GetFullTitle("uk", nationalVersion),
			ID:                1,
			MajorityID:        item.Data.MajorityID,
			ObjID:             item.Data.ObjID,
			ObjType:           item.Data.ObjType,
			Order:             0,
			PointKey:          item.Data.PointKey,
			// TODO: В оригинале тут 'region_id': item.get('zone_id'), возможно, из-за копипаста.
			RegionID:        item.Data.RegionID,
			Slug:            item.Data.Slug,
			TType:           item.Data.TTypeCode,
			Title:           item.Data.Titles[lang],
			TitleEN:         item.Data.Titles["en"],
			TitleRU:         item.Data.Titles["ru"],
			TitleUK:         item.Data.Titles["uk"],
			ZoneID:          item.Data.ZoneID,
			PopularTitle:    item.Data.PopularTitles[lang],
			StationType:     item.Data.StationTypes[lang],
			SettlementTitle: item.Data.SettlementTitles[lang],
			RegionTitle:     item.Data.RegionTitles[lang],
			CountryTitle:    item.Data.GetCountryTitle(lang, nationalVersion),
		}
		if !item.IsPrefix {
			hasNotPrefix = true
		}
	}

	resp, err := models.DictStyleResponse{Suggests: result, TotalFound: len(result), HasExactMatch: hasNotPrefix}.MarshalJSON()
	if err != nil {
		logger.Errorf("Error while marshalling error: %+v", err)
		return ""
	}
	return string(resp)

}

func newStyleFormat(items []models.MainSearchResult, lang, nationalVersion string) string {
	respItems := make([]models.NewStyleStruct, len(items))
	for i, item := range items {
		respItems[i] = models.NewStyleStruct{
			ObjID:        item.Data.ObjID,
			ObjType:      item.Data.ObjType,
			FullTitle:    item.GetFullTitle(lang, nationalVersion),
			Title:        item.Data.Titles[lang],
			Slug:         item.Data.Slug,
			Settlement:   item.Data.SettlementTitles[lang],
			Region:       item.Data.RegionTitles[lang],
			Country:      item.Data.GetCountryTitle(lang, nationalVersion),
			PopularTitle: item.Data.PopularTitles[lang],
			StationType:  item.Data.StationTypes[lang],
			PointKey:     item.Data.PointKey,
		}
	}

	jsonItems, err := models.NewStyleResponse{Items: respItems}.MarshalJSON()
	if err != nil {
		logger.Errorf("Error while marshalling error: %+v", err)
		return "{}"
	}
	return string(jsonItems)
}

var formatterMapping = map[string]responseFormatter{
	"old": oldStyleFormat,
	"all": allStyleFormat,
	"new": newStyleFormat,
}

func getFormatter(format string) responseFormatter {
	if res, ok := formatterMapping[format]; ok {
		return res
	}
	return allStyleFormat
}
