package web

import (
	"fmt"
	"net/http"
	"net/url"
	"os"
	"strconv"
	"strings"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/rasp/suggests/logger"
	"a.yandex-team.ru/travel/rasp/suggests/models"
	"a.yandex-team.ru/travel/rasp/suggests/search"
	"a.yandex-team.ru/travel/rasp/suggests/utils"
)

const (
	defaultLimitOfResponses = 10
)

var (
	waterTTypes           = []string{"river", "sea", "water"}
	titleLangs            = []string{"ru", "en", "uk"}
	titleNationalVersions = []string{"ru", "ua"}
)

// Настройки web-приложения.
type App struct {
	searchEngine *search.SearchCreator
	idConverter  utils.IDConverter
	objectData   *models.ObjectData
}

func NewWebApp(sc *search.SearchCreator, idConv utils.IDConverter, data *models.ObjectData) App {
	return App{sc, idConv, data}
}

func (a *App) Update(sc *search.SearchCreator, idConv utils.IDConverter, data *models.ObjectData) {
	a.searchEngine = sc
	a.idConverter = idConv
	a.objectData = data
}

func renderMarshallingError() string {
	resp := models.Response{Result: []models.ResponseEntity{}, Errors: []models.ErrorMessage{{Message: "Bad data."}}}
	jsonResp, err := resp.MarshalJSON()
	if err != nil {
		logger.Errorf("Error while marshalling error: %+v", err)
	}
	return string(jsonResp)
}

func renderResponse(result []models.ResponseEntity, errors []models.ErrorMessage) string {
	resp := models.Response{Result: result, Errors: errors}
	jsonResp, err := resp.MarshalJSON()
	if err != nil {
		logger.Errorf("Error while marshalling json: %+v", err)
		return renderMarshallingError()
	}
	return string(jsonResp)
}

func (a *App) titleDataToResponse(data []models.MainSearchResult) []models.ResponseEntity {
	var result []models.ResponseEntity
	for _, item := range data {
		info := a.idConverter.LocalIDToDB(item.Data.ID)
		name := a.objectData.ObjectsData[item.Data.ID].Titles["ru"]
		result = append(result, models.ResponseEntity{ID: info.ID, Type: info.Type, Name: name})
	}
	return result
}

func getParam(query string, key string) (string, bool) {
	params, _ := url.ParseQuery(query)
	if keyParams, ok := params[key]; ok && len(keyParams) > 0 {
		return keyParams[0], true
	}
	return "", false
}

func getIntParam(query string, key string, defaultValue int) int {
	if res, ok := getParam(query, key); ok {
		if resInt, err := strconv.Atoi(res); err == nil {
			return resInt
		}
	}
	return defaultValue
}

func getStringParam(query string, key string, defaultValue string) string {
	if res, ok := getParam(query, key); ok {
		return res
	}
	return defaultValue
}

func (a *App) SearchItems(suggestType string, defaultTType string, w http.ResponseWriter, r *http.Request) {
	request, ok := getParam(r.URL.RawQuery, "part")
	if !ok {
		if request, ok = getParam(r.URL.RawQuery, "query"); !ok {
			request = ""
		}
	}
	otherPoint, _ := getParam(r.URL.RawQuery, "other_point")
	var from *models.DBLocationInfo
	if len(otherPoint) > 0 {
		if id, err := strconv.Atoi(otherPoint[1:]); err == nil {
			switch otherPoint[0] {
			case 's':
				from = &models.DBLocationInfo{ID: id, Type: "station"}
			case 'c':
				from = &models.DBLocationInfo{ID: id, Type: "settlement"}
			}
		}
	}
	callback, hasCallback := getParam(r.URL.RawQuery, "callback")
	region := getIntParam(r.URL.RawQuery, "client_city", -1)
	limit := getIntParam(r.URL.RawQuery, "limit", defaultLimitOfResponses)
	suggestFormat := getStringParam(r.URL.RawQuery, "format", "all")
	nationalVersion := getStringParam(r.URL.RawQuery, "national_version", "ru")
	if !utils.StringInSlice(nationalVersion, titleNationalVersions) {
		nationalVersion = "ru"
	}
	lang := getStringParam(r.URL.RawQuery, "lang", "ru")
	if !utils.StringInSlice(lang, titleLangs) {
		lang = "ru"
	}
	ttypes := []string{"all"}
	if suggestType == "suburban" {
		ttypes = []string{"suburban"}
	} else if suggestType == "common" {
		ttypes = []string{"all"}
	} else {
		if defaultTType == "" {
			defaultTType = getStringParam(r.URL.RawQuery, "t_type_code", "")
		}
		if utils.StringInSlice(defaultTType, waterTTypes) {
			defaultTType = "water"
		}
		if defaultTType != "" {
			ttypes = []string{defaultTType}
		}
	}
	suggestFilterType := ""
	if suggestType == "city" {
		suggestFilterType = "settlement"
	} else if suggestType == "station" {
		suggestFilterType = "station"
	} else if suggestType == "pointkey" {
		suggestFilterType = "pointkey"
	}
	logger.Infof("Handling request (request: %v, limit: %v, region: %v, from: %v, ttypes: %v, lang: %v, NV: %v", request, limit, region, from, ttypes, lang, nationalVersion)
	result, err := a.searchEngine.Find(
		request,
		limit,
		region,
		from,
		ttypes,
		lang,
		nationalVersion,
		suggestFilterType,
		a.idConverter,
		logger.With(log.String("request", request)),
	)
	if err != nil {
		logger.Errorf("Searching for %s failed: %+v.", request, err)
		_, _ = fmt.Fprintln(w, renderResponse([]models.ResponseEntity{}, []models.ErrorMessage{{Message: "Bad data"}}))
		return
	}

	w.Header().Set("X-Content-Type-Options", "nosniff")
	w.Header().Set("X-Frame-Options", "SAMEORIGIN")
	if origin := r.Header.Get("Origin"); utils.IsValidOrigin(origin) || os.Getenv("YENV_TYPE") == "testing" {
		w.Header().Set("Access-Control-Allow-Origin", origin)
	}

	if hasCallback {
		if !utils.IsValidPJSON(callback) {
			w.WriteHeader(http.StatusMethodNotAllowed)
			_, _ = fmt.Fprintln(w, "Callback name should be 64 symbols long at max and contain only [a-zA-Z0-9_] symbols.")
			return
		}

		w.Header().Set("Content-Type", "application/javascript; charset=utf-8")
		w.Header().Set("Content-Disposition", "attachment; filename=json.txt")

		formatter := getFormatter(suggestFormat)
		_, _ = fmt.Fprintf(w, "/**/%s(%s);", callback, formatter(result, lang, nationalVersion))
	} else {
		w.Header().Set("Content-Type", "application/json; charset=utf-8")
		formatter := getFormatter(suggestFormat)
		_, _ = fmt.Fprintln(w, formatter(result, lang, nationalVersion))
	}
}

func (a *App) SearchFactory(suggestType string, defaultTType string) func(http.ResponseWriter, *http.Request) {
	logger.Info("Creating searchClosure")
	searchClosure := func(w http.ResponseWriter, r *http.Request) {
		a.SearchItems(suggestType, defaultTType, w, r)
	}
	return searchClosure
}

func (a *App) NewSearch(w http.ResponseWriter, r *http.Request) {
	request := getStringParam(r.URL.RawQuery, "text", "")
	limit := getIntParam(r.URL.RawQuery, "limit", defaultLimitOfResponses)
	region := getIntParam(r.URL.RawQuery, "region", -1)
	lang := getStringParam(r.URL.RawQuery, "lang", "ru")
	if !utils.StringInSlice(lang, titleLangs) {
		lang = "ru"
	}
	nationalVersion := getStringParam(r.URL.RawQuery, "national_version", "ru")
	if !utils.StringInSlice(nationalVersion, titleNationalVersions) {
		nationalVersion = "ru"
	}
	otherPoint := getStringParam(r.URL.RawQuery, "other_point", "")
	var from *models.DBLocationInfo
	if len(otherPoint) > 0 {
		data := strings.Split(otherPoint, ":")
		if len(data) == 2 {
			if id, err := strconv.Atoi(data[1]); err == nil && (data[0] == "station" || data[0] == "settlement") {
				from = &models.DBLocationInfo{ID: id, Type: data[0]}
			}
		}
	}
	var ttypes []string
	if ttypesStr, ok := getParam(r.URL.RawQuery, "transportTypes"); ok {
		rawTTypes := strings.Split(ttypesStr, ",")
		ttypes = make([]string, len(rawTTypes))
		for i := range rawTTypes {
			ttypes[i] = strings.TrimSpace(rawTTypes[i])
		}
	} else {
		ttypes = []string{"all"}
	}

	result, err := a.searchEngine.Find(
		request,
		limit,
		region,
		from,
		ttypes,
		lang,
		nationalVersion,
		"",
		a.idConverter,
		logger.With(log.String("request", request)),
	)
	if err != nil {
		logger.Errorf("web.web:Search: Searching for %s failed: %v", request, err)
		_, _ = fmt.Fprintln(w, renderResponse([]models.ResponseEntity{}, []models.ErrorMessage{{Message: "Bad data"}}))
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=utf-8")
	formatter := getFormatter("new")
	_, _ = fmt.Fprintln(w, formatter(result, lang, nationalVersion))
}

func Ping(w http.ResponseWriter, r *http.Request) {
	w.WriteHeader(http.StatusOK)
	_, _ = fmt.Fprintln(w, "ok")
}
