# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from collections import defaultdict

from django.contrib.contenttypes.models import ContentType

from common.models.geo import StationCode, SettlementCode, CodeSystem, PointSynonym
from common.precache.backend import precache


logger = logging.getLogger('generate')


def precache_django_models():
    precache()


class Precache(object):
    def __init__(self):
        self.is_precached = False

    def precache(self):
        self.is_precached = True

    def get(self, obj):
        return self.get_if_precached(obj) if self.is_precached else self.get_if_not_precached(obj)

    def get_if_precached(self, obj):
        raise NotImplementedError

    def get_if_not_precached(self, obj):
        raise NotImplementedError


class CodePrecache(Precache):
    model = None

    def __init__(self, systems, verbose=False):
        super(CodePrecache, self).__init__()

        self.codes = defaultdict(dict)
        self.allowed_systems = list(CodeSystem.objects.filter(code__in=systems))
        self.verbose = verbose

    def precache(self):
        if self.verbose:
            logger.info('{}.precache started'.format(self.__class__.__name__))

        for code in self.model.objects.filter(system__in=self.allowed_systems):
            self.codes[code.station][code.system.code] = code

        if self.verbose:
            logger.info('{}.precache finished'.format(self.__class__.__name__))

        super(CodePrecache, self).precache()

    def get_if_precached(self, obj):
        return self.codes.get(obj, {})


class StationCodePrecache(CodePrecache):
    model = StationCode

    def get_if_not_precached(self, obj):
        return {
            code.system.code: code
            for code in self.model.objects.filter(system__in=self.allowed_systems, station=obj)
        }


class SettlementCodePrecache(CodePrecache):
    model = SettlementCode

    def get_if_not_precached(self, obj):
        return {
            code.system.code: code
            for code in self.model.objects.filter(system__in=self.allowed_systems, settlement=obj)
        }


class SynonymsPrecache(Precache):
    def __init__(self, models):
        super(SynonymsPrecache, self).__init__()
        self.models = models
        self.cache = defaultdict(lambda: defaultdict(list))

    def precache(self):
        for model in self.models:
            content_type = ContentType.objects.get_for_model(model)
            synonyms = PointSynonym.objects.filter(content_type=content_type)
            for synonym in synonyms:
                self.cache[model][synonym.object_id].append(synonym)

        super(SynonymsPrecache, self).precache()

    def get_if_precached(self, obj):
        try:
            return self.cache[obj.__class__][obj.id]
        except KeyError:
            return []

    def get_if_not_precached(self, obj):
        return list(PointSynonym.objects.filter(
            content_type=ContentType.objects.get_for_model(obj.__class__),
            object_id=obj.id))
