# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import multiprocessing
import sys
import traceback
from contextlib import contextmanager

from django.db import connection


logger = logging.getLogger('generate')


@contextmanager
def get_pool(pool_size):
    connection.close()  # иначе воркеры в пуле пошарят соединение через fork

    pool = multiprocessing.Pool(pool_size)
    try:
        yield pool
    finally:
        pool.close()
        pool.join()


def error_catcher((func, args)):
    try:
        return func(args)
    except BaseException:
        raise Exception("".join(traceback.format_exception(*sys.exc_info())))


def generate_parallel(func, workers_data, pool_size):
    logger.info('{} on {} processes'.format(func, pool_size))
    with get_pool(pool_size) as pool:
        for result in pool.imap_unordered(error_catcher, ((func, args) for args in workers_data.items())):
            yield result


def retrieve_ids(model, __limit=None, exclude_t_type_ids=None, **kwargs):
    query_set = model.objects
    if exclude_t_type_ids:
        query_set = query_set.exclude(thread__t_type_id__in=exclude_t_type_ids)

    return [o['id'] for o in query_set.filter(**kwargs).values('id')[:__limit]]


def raw_sql(query):
    c = connection.cursor()
    c.execute(query)
    return c
