# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import os
import os.path

import msgpack

from travel.rasp.suggests_tasks.suggests.objects_utils import ObjIdConverter
from travel.rasp.suggests_tasks.suggests.utils import mkdir, print_run_time


class Storage(object):
    def __init__(self, data_path, logger=None):
        self.data_path = data_path
        self.logger = logger

        self.id_conv_file = 'idconverter.msgpack'
        self.t_types_file = 'ttypes.msgpack'
        self.stat_file = 'stat.msgpack'
        self.stat_converted_file = 'stat_converted.msgpack'
        self.objs_data_file = 'objs_data.msgpack'

    def __eq__(self, other):
        return self.data_path == other.data_path

    def _save_data(self, filename, data):
        mkdir(self.data_path)
        data_file = os.path.join(self.data_path, filename)
        with open(data_file, 'w') as f:
            with print_run_time('saving: {}'.format(filename), logger=self.logger):
                msgpack.dump(data, f, encoding='utf8')

    def _load_data(self, filename):
        with print_run_time('loading: {}'.format(filename), print_before=False, logger=self.logger):
            f = open(os.path.join(self.data_path, filename))
            return msgpack.load(f, encoding='utf8', use_list=False)

    def load_id_converter(self, freeze, can_create=True):
        try:
            id_conv_data = self._load_data(self.id_conv_file)
        except IOError:
            if not can_create:
                raise

            id_converter = None
        else:
            id_converter = ObjIdConverter.from_dict(id_conv_data, freeze=freeze)

        if not id_converter:
            id_converter = ObjIdConverter()

        return id_converter

    def save_id_converter(self, id_converter):
        self._save_data(self.id_conv_file, id_converter.to_dict())

    def load_ttypes(self):
        return self._load_data(self.t_types_file)

    def save_ttypes(self, ttypes):
        """
        :param ttypes:
        {
            'stations': {
                obj_id: [ttype, ...],
                ...
            },
            'settlements': {
                obj_id: [ttype, ...],
                ...
            }
        }
        """
        self._save_data(self.t_types_file, ttypes)

    def load_stat(self):
        return self._load_data(self.stat_file)

    def save_stat(self, stat):
        self._save_data(self.stat_file, stat)

    def load_stat_converted(self):
        return self._load_data(self.stat_converted_file)

    def save_stat_converted(self, stat_converted):
        """
        :param stat_converted:
        {
            'by_obj': {}
            'routes': {},
            'source_tables': [],
        }
        """
        self._save_data(self.stat_converted_file, stat_converted)

    def load_objs_data(self):
        # TODO: titles lists to sets

        return self._load_data(self.objs_data_file)

    def save_objs_data(self, objs_data):
        """
        :param objs_data:
        {
            'objects_data': {},
            'titles': {
                lang: {
                    ttype: {
                        title: {(obj_id1, is_prefix), (obj_id2, is_prefix), ...},
                        ...
                    },
                    ...
                },
                ...
            },
            'station_prefixes': {},
            'db_info': [],
        }
        """

        # msgpack не сохраняет set, преобразовываем в list
        titles = objs_data['titles']
        for lang, ttypes_data in titles.items():
            for t_type, titles_data in ttypes_data.items():
                for title, obj_ids in titles_data.items():
                    titles_data[title] = list(obj_ids)

        self._save_data(self.objs_data_file, objs_data)
