# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import errno
import logging
import os
from contextlib import contextmanager
from copy import copy
from hashlib import md5
from itertools import izip_longest
from time import time


def get_stdout_logger():
    logger = logging.getLogger(md5().hexdigest())
    logger.level = logging.DEBUG
    logger.propagate = False

    handler = logging.StreamHandler()
    logger.addHandler(handler)
    handler.setFormatter(logging.Formatter('%(message)s'))

    return logger


stdout_logger = get_stdout_logger()


class Enumerator(object):
    def __init__(self, **kwargs):
        log = kwargs.get('log')
        if log is None:
            kwargs['logger'] = stdout_logger
            kwargs['log_level'] = stdout_logger.level

        self.default_params = kwargs

        self.count = 0

    def __call__(self, iterable, **kwargs):
        self.count = 0

        params = copy(self.default_params)
        params.update(kwargs)

        logger = params['logger']
        log_level = params.get('log_level', logging.INFO)

        def prnt(*args):
            logger.log(log_level, ' '.join(map(str, args)))

        disable = params.get('disable', False)
        each = params.get('each', 1000)
        each_time = params.get('each_time', True)
        start = params.get('start', 0)
        yield_num = params.get('yield_num', False)
        title = params.get('title', u'')

        if disable:
            for v in iterable:
                yield v

        start_time = time()
        for i, v in enumerate(iterable, start=start):
            if not (i % each) and i != 0:
                msg = unicode(i)
                if each_time:
                    msg += u' {:.4}'.format(time() - start_time)
                    start_time = time()

                prnt(msg)

            self.count += 1
            if yield_num:
                yield i, v
            else:
                yield v

        msg = unicode(self.count)
        if each_time:
            msg += u' {:.4}'.format(time() - start_time)
        prnt(msg)

        if not each_time:  # при each_time и так пишется total
            if title:
                prnt(u'total {} {}'.format(title, self.count))
            else:
                prnt(u'total {}'.format(self.count))


enumer = Enumerator()


@contextmanager
def print_run_time(title=u'', print_before=True, logger=None, log_level=logging.INFO):
    start = time()

    logger = stdout_logger if logger is None else logger

    def prnt(*args):
        logger.log(log_level, u' '.join(map(unicode, args)))

    if print_before:
        prnt(u'{} -> started'.format(title))

    run_time = None

    def time_getter():
        return run_time

    try:
        yield time_getter
    finally:
        run_time = time() - start
        prnt(u'{} -> done in {:.4}'.format(title, run_time))


def mkdir(path):
    try:
        os.makedirs(path)
    except OSError as ex:
        if ex.errno == errno.EEXIST and os.path.isdir(path):
            pass
        else:
            raise


def split_values(values, slices_num):
    """ Разбиваем последовательность values на slices_num частей """
    if not values:
        return {i: [] for i in range(slices_num)}

    nothing = object()
    result = {}
    it = iter(values)
    for zipped_values in izip_longest(*([it] * slices_num), fillvalue=nothing):
        for i, v in enumerate(zipped_values):
            slice_values = result.setdefault(i, [])
            if v is not nothing:
                slice_values.append(v)

    return result


def get_multi(d, keys, default=None):
    """ 'get' for hierarchy of dicts

        d = {1: {2: {3: 'a'}}}
        get_multi(d, [1, 2, 3], default='nothing')  # --> a
        get_multi(d, [1, 42, 3], default='nothing')  # --> nothing
    """
    current = d
    for key in keys[:-1]:
        current = current.get(key, {})

    return current.get(keys[-1], default)


def set_multi(d, keys, value):
    """ set value in hierarchy of dicts

        d = {}
        set_multi(d, [1, 2, 3], 'a')  # --> d == {1: {2: {3: 'a'}}}
    """
    current = d
    for key in keys[:-1]:
        current = current.setdefault(key, {})

    current[keys[-1]] = value
