# coding: utf-8

"""Список всех станций с кодами yandex, esr для ручки апи (https://st.yandex-team.ru/RASPAPI-696)"""

import json  # noqa
import logging
from collections import defaultdict

from django.db.models import Prefetch
from lxml import etree

from common.db.mds.clients import mds_s3_common_client
from common.models.geo import Station, ExternalDirectionMarker, StationCode, CodeSystem
from common.models_utils import fetch_related
from common.utils.lock import lock
from travel.rasp.library.python.common23.logging import log_run_time
from travel.rasp.library.python.common23.logging.scripts import script_context
from travel.rasp.tasks.utils import chunker

log = logging.getLogger('api_public_stations_list')


FILE_NAME = 'stations_list'
NATIONAL_VERSION_AND_LANG_FILES = [
    ('ru', 'ru'),
    ('ua', 'uk'),
    ('ua', 'ru'),
]


def get_station_direction(station):
    markers = station.externaldirectionmarker_set.all()
    if markers:
        return markers[0].external_direction


def station_2_dict(station, station_esr_codes, lang):
    json_station = {
        'title': station.L_title(lang=lang),
        'longitude': station.longitude or '',
        'latitude': station.latitude or '',
        'transport_type': station.t_type.code,
        'station_type': station.station_type.code,
        'codes': {'yandex_code': station.point_key}
    }

    direction = get_station_direction(station)
    json_station['direction'] = direction.L_title(lang=lang) if direction else ''

    esr_code = station_esr_codes.get(station.id)
    if esr_code:
        json_station['codes']['esr_code'] = esr_code

    return json_station


def generate_xml(stations_dict, points_attrs):
    def add_element_attrs(point_element, point_key):
        if point_key == 'unknown':
            etree.SubElement(point_element, 'title')
            etree.SubElement(point_element, 'codes')
            return
        etree.SubElement(point_element, 'title').text = points_attrs[point_key]['title']
        code_el = etree.Element('codes')
        etree.SubElement(code_el, 'yandex_code').text = point_key
        point_element.append(code_el)

    tree = etree.Element('response')
    for country, regions in stations_dict.items():
        country_el = etree.Element('country')
        add_element_attrs(country_el, country)
        for region, settlements in regions.items():
            region_el = etree.Element('region')
            add_element_attrs(region_el, region)
            for settlement, stations in settlements.items():
                settlement_el = etree.Element('settlement')
                add_element_attrs(settlement_el, settlement)
                for station in stations:
                    station_el = etree.Element('station')
                    etree.SubElement(station_el, 'title').text = station['title']
                    etree.SubElement(station_el, 'longitude').text = str(station['longitude'])
                    etree.SubElement(station_el, 'latitude').text = str(station['latitude'])
                    etree.SubElement(station_el, 'transport_type').text = station['transport_type']
                    etree.SubElement(station_el, 'station_type').text = station['station_type']

                    station_direction = station.get('direction')
                    if station_direction:
                        etree.SubElement(station_el, 'direction').text = station_direction

                    codes_el = etree.Element('codes')
                    for system, code in station['codes'].items():
                        etree.SubElement(codes_el, system).text = code
                    station_el.append(codes_el)

                    settlement_el.append(station_el)
                region_el.append(settlement_el)
            country_el.append(region_el)
        tree.append(country_el)
    return tree


def generate_json(stations_dict, points_attrs):
    def add_point_attrs(point_key, kwargs):
        if point_key == 'unknown':
            attrib = {'codes': {}, 'title': ''}
        else:
            attrib = {'codes': {'yandex_code': point_key}, 'title': points_attrs[point_key]['title']}
        kwargs.update(attrib)
        return kwargs

    json_stations = {'countries': []}
    for country, regions in stations_dict.items():
        regions_list = []
        for region, settlements in regions.items():
            settlements_list = [add_point_attrs(settlement, {'stations': stations})
                                for settlement, stations in settlements.items()]
            regions_list.append(add_point_attrs(region, {'settlements': settlements_list}))

        json_stations['countries'].append(add_point_attrs(country, {'regions': regions_list}))
    return json_stations


def generate_stations_dict(national_version, lang):
    stations = Station.objects.all().values_list('id', flat=True)
    stations_by_points = defaultdict(lambda: defaultdict(lambda: defaultdict(list)))
    points_attrs = defaultdict(lambda: dict)

    for station_chunk_ids in chunker(stations, 15000):
        dir_prefetch = Prefetch('externaldirectionmarker_set',
                                queryset=ExternalDirectionMarker.objects.select_related('external_direction'))
        station_chunk = Station.objects.filter(id__in=station_chunk_ids).prefetch_related(dir_prefetch)
        fetch_related(station_chunk, 'region', 'country', 'settlement', 'district', 't_type', 'station_type')

        station_esr_codes = StationCode.StationCodeByStationIdGetter(
            CodeSystem.objects.get(code='esr'),
            station_ids=station_chunk_ids
        )

        for station in station_chunk:
            real_country = station.translocal_country(national_version=national_version)
            country = real_country.point_key if real_country else "unknown"
            region = station.region.point_key if station.region else "unknown"
            settlement = station.settlement.point_key if station.settlement else "unknown"

            for obj in [real_country, station.region, station.settlement]:
                if obj and not points_attrs.get(obj.point_key):
                    points_attrs[obj.point_key] = {'title': obj.L_title(lang=lang)}

            stations_by_points[country][region][settlement].append(station_2_dict(station, station_esr_codes, lang))

    return stations_by_points, points_attrs


def save_data_mds(data, national_version, lang, file_format):
    key = u'api_public/stations_list_{}_{}.{}'.format(national_version, lang, file_format)
    mds_s3_common_client.save_data(data=data, key=key)


def generate_stations_list():
    for (national_version, lang) in NATIONAL_VERSION_AND_LANG_FILES:
        with log_run_time('generate_stations_dict', logger=log):
            stations, points_attrs = generate_stations_dict(national_version, lang)
        with log_run_time('generate_xml', logger=log):
            stations_xml = generate_xml(stations, points_attrs)
        with log_run_time('generate_json', logger=log):
            stations_json = generate_json(stations, points_attrs)

        with log_run_time('save_to_mds', logger=log):
            data_json_dump = json.dumps(stations_json, ensure_ascii=False, encoding='utf-8')
            data_xml = etree.tostring(stations_xml, encoding='utf8')
            save_data_mds(data_json_dump, national_version, lang, 'json')
            save_data_mds(data_xml, national_version, lang, 'xml')


def run():
    with lock('lock_stations_list', 'api_public'), script_context('api_public_stations_list'):
        try:
            generate_stations_list()
        except Exception as ex:
            log.exception(u'failed: {}'.format(repr(ex)))
            raise


if __name__ == '__main__':
    run()
