# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from datetime import datetime, timedelta

import argparse
import logging
from typing import AnyStr

from django.conf import settings

from yt.wrapper import YtClient, ypath_join


PATHES = [
    'rasp-min-prices-by-routes',
    'rasp-min-prices-some-days',
    'rasp-min-train-prices-by-route'
]
DATE_PATTERN = '%Y-%m-%d'
DEFAULT_DAYS_AGO = 30
DEFAULT_REMOVE_OLD_TABLES_DAYS = 3

log = logging.getLogger(__name__)


class MinPricesYtCleaner:
    def __init__(
        self,
        environment,            # type: AnyStr
        yt_token,               # type: AnyStr
        yt_proxy,               # type: AnyStr
        yt_root_path,           # type: AnyStr
        days_ago,               # type: int
        remove_old_tables_days  # type: int
    ):
        self.yt_client = YtClient(proxy=yt_proxy, token=yt_token)
        self.base_path = ypath_join(yt_root_path, environment)
        self.days_ago = days_ago
        self.remove_old_tables_days = remove_old_tables_days

    def _remove_table(self, path):
        # type: (AnyStr) -> None

        if self.yt_client.exists(path):
            log.info('removing table {}'.format(path))
            self.yt_client.remove(path)
        else:
            log.warning('{} not exists'.format(path))

    def run(self):
        start_date = datetime.today() - timedelta(days=self.days_ago)

        for path in PATHES:
            log.info('--- cleaning {} ---'.format(path))
            directory_path = ypath_join(self.base_path, path)
            for days_delta in range(self.remove_old_tables_days):
                day = start_date + timedelta(days=days_delta)
                day_path = ypath_join(directory_path, day.strftime(DATE_PATTERN))
                self._remove_table(day_path)


def run(days_ago=DEFAULT_DAYS_AGO, remove_old_tables_days=DEFAULT_REMOVE_OLD_TABLES_DAYS):
    # type: (int, int) -> None

    MinPricesYtCleaner(
        settings.YANDEX_ENVIRONMENT_TYPE,
        settings.YT_TOKEN,
        settings.YT_PROXY,
        settings.YT_ROOT_PATH,
        days_ago,
        remove_old_tables_days
    ).run()


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--days-ago', dest='from_day', type=int, default=DEFAULT_DAYS_AGO)
    parser.add_argument('--remove-old-tables-days', dest='to_day', type=int, default=DEFAULT_REMOVE_OLD_TABLES_DAYS)
    args = parser.parse_args()
    run(args.days_ago, args.remove_old_tables_days)
