# coding: utf-8

import json
import logging
import os
import tempfile
from datetime import datetime, timedelta
from functools import reduce

import yt.wrapper

log = logging.getLogger(__name__)

STATION_MAP_FILE_NAME = os.path.join(tempfile.gettempdir(), 'station_map.json')


class YtMinTrainPrice:
    def __init__(self, station_settlement_map, log_search_depth, trip_days_count,
                 source_dir, destination_table, tmp_dir,
                 token, proxy):
        self.station_settlement_map = station_settlement_map
        self.log_search_depth = log_search_depth
        self.trip_days_count = trip_days_count
        self.source_dir = source_dir
        self.destination_table = destination_table
        self.tmp_dir = tmp_dir
        self.token = token
        self.proxy = proxy
        self.tariffs_table = []
        self.trip_days = set()
        self.fields = ['date_forward', 'object_from_id', 'object_to_id', 'price']

    def yt_init(self):
        yt.wrapper.update_config(
            {
                "token": self.token,
                "proxy": {'url': self.proxy}
            }
        )

    def yt_find_source_tables(self):
        source_tables = []
        for day_ago in range(1, self.log_search_depth + 1):
            table_path = os.path.join(self.source_dir,
                                      (datetime.today() - timedelta(days=day_ago)).strftime("%Y-%m-%d"))
            if yt.wrapper.exists(table_path):
                source_tables.append(table_path)
            else:
                log.warning('Path %s does not exist', table_path)
        return source_tables

    def start(self):
        self.yt_init()
        source_tables = self.yt_find_source_tables()
        self.trip_days = {(datetime.today() + timedelta(days=day)).strftime("%Y-%m-%d")
                          for day in range(0, self.trip_days_count)}
        log.info('source tables: %s', str(source_tables))
        if not source_tables:
            log.warning('Skip map-reduce. No source tables')
            return
        yt.wrapper.mkdir(self.tmp_dir, recursive=True)
        log.info('map-reduce run')
        yt.wrapper.run_map_reduce(
            self.map_source,
            self.reduce_min_price,
            source_tables,
            self.destination_table,
            reduce_by=['settlement_from_id', 'settlement_to_id', 'object_from_id', 'object_to_id', 'date_forward']
        )
        log.info('run done')

    def map_source(self, row):
        if all([field in row for field in self.fields]):
            if row['type'] == 'train':
                if row['date_forward'] in self.trip_days:
                    settlement_from_id = self.station_settlement_map.get(row['object_from_id'])
                    settlement_to_id = self.station_settlement_map.get(row['object_to_id'])
                    if settlement_from_id and settlement_to_id:
                        row['price'] = float(row['price'])
                        res = {'settlement_from_id': settlement_from_id, 'settlement_to_id': settlement_to_id}
                        for f in self.fields:
                            res[f] = row[f]
                        yield res

    def reduce_min_price(self, _key, recs):
        best_price = reduce(lambda a, b:
                            a if float(a['price']) < float(b['price']) else b,
                            recs)
        yield best_price


if __name__ == "__main__":
    import argparse
    parser = argparse.ArgumentParser()
    parser.add_argument('--log_search_depth', type=int)
    parser.add_argument('--trip_days_count', type=int)
    parser.add_argument('--source_dir')
    parser.add_argument('--destination_table')
    parser.add_argument('--tmp_dir')
    parser.add_argument('--token')
    parser.add_argument('--proxy')
    args = parser.parse_args()

    with open(STATION_MAP_FILE_NAME, "r") as f:
        station_map = json.loads(f.read())

    runner = YtMinTrainPrice(
        station_map,
        args.log_search_depth,
        args.trip_days_count,
        args.source_dir,
        args.destination_table,
        args.tmp_dir,
        args.token,
        args.proxy
    )
    runner.start()
