# coding: utf-8

import argparse
import logging
import os
import sys
from datetime import datetime, timedelta

import yt.wrapper
from django.conf import settings
from django.utils.lru_cache import lru_cache

from common.data_api.min_prices.api import min_price_storage
from common.data_api.yt.instance import update_yt_wrapper_config
from common.models.geo import Station, Station2Settlement
from common.utils.lock import lock
from travel.rasp.library.python.common23.logging import log_run_time
from travel.rasp.library.python.common23.logging.scripts import script_context


log = logging.getLogger('yt_to_mongo')


@lru_cache(maxsize=None)
def get_settlement_ids(object_id):
    if not object_id:
        return {0}
    station = Station.objects.filter(id=object_id).first()
    if station and station.settlement:
        settlement_ids = {int(station.settlement.id)}
        for settlement_id in Station2Settlement.objects.filter(station_id=object_id).values_list('id', flat=True):
            settlement_ids.add(int(settlement_id))
        return settlement_ids
    return {0}


def erase_tuple(str):
    if str == 'StationTuple':
        return 'Station'
    if str == 'SettlementTuple':
        return 'Settlement'
    return str


def wrong_encoding(price):
    for key, value in price.items():
        try:
            key.decode('utf-8')
            value.decode('utf-8')
        except UnicodeDecodeError:
            return True
    return False


def prepare_data(yt_min_prices):
    for price in yt_min_prices:
        if price['class'] == 'None':
            price['class'] = 'platzkart'
        elif price['class'] == 'economy-bus':
            price['class'] = 'economy'
        elif price['class'] in ('first', 'business'):
            continue

        price['object_from_type'] = erase_tuple(price['object_from_type'])
        price['object_to_type'] = erase_tuple(price['object_to_type'])

        price['{}_from_id'.format(price['object_from_type'])] = price['object_from_id']
        price['{}_to_id'.format(price['object_to_type'])] = price['object_to_id']

        if wrong_encoding(price):
            continue

        for settlement_from_id in get_settlement_ids(price['Station_from_id'] if 'Station_from_id' in price else None):
            for settlement_to_id in get_settlement_ids(price['Station_to_id'] if 'Station_to_id' in price else None):
                if settlement_from_id > 0:
                    price['Settlement_from_id'] = settlement_from_id
                if settlement_to_id > 0:
                    price['Settlement_to_id'] = settlement_to_id

                yield price


def upload_min_prices(file_name, days_ago, remove_old_prices_days):
    table_path = os.path.join(settings.YT_ROOT_PATH, settings.YANDEX_ENVIRONMENT_TYPE, file_name,
                              (datetime.today() - timedelta(days=days_ago)).strftime('%Y-%m-%d'))
    log.info('table_path: %s', table_path)

    update_yt_wrapper_config()

    if not yt.wrapper.exists(table_path):
        log.error('Path %s does not exist', table_path)
        sys.exit(1)

    log.info('Rows in mongo initially: %d', min_price_storage.count())

    with log_run_time('deleting_old_prices_from_mongo', logger=log):
        min_price_storage.remove_old_rows(days_ago=remove_old_prices_days)
    log.info('Rows after deleting 1: %d', min_price_storage.count())

    with log_run_time('inserting_min_prices_to_mongo', logger=log):
        prepared_data = prepare_data(yt.wrapper.read_table(table_path, format='dsv', raw=False))
        min_price_storage.save_many(prepared_data)
    log.info('Rows after inserting: %d', min_price_storage.count())

    with log_run_time('deleting_old_prices_from_mongo', logger=log):
        min_price_storage.remove_old_rows(days_ago=100)
    log.info('Rows after deleting 2: %d', min_price_storage.count())


def run(file_name, days_ago, remove_old_prices_days):
    with lock('min_prices_yt_to_mongo'), script_context('min_prices_yt_to_mongo'):
        upload_min_prices(file_name, days_ago, remove_old_prices_days)


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '-f', '--file_name', dest='file_name', type=str, default='rasp-min-prices-by-routes',
        help=u'папка, откуда брать файл с ценами, '
             u'цены за 3 дня - ./yt_to_mongo.py -d 0 -f rasp-min-prices-some-days')
    parser.add_argument('-d', '--days_ago', dest='days_ago', type=int, default=1, help=u'1 - вчера, 2 - позавчера')
    parser.add_argument('-r', '--remove_old_prices_days', dest='remove_old_prices_days', type=int, default=3,
                        help=u'0 - удалить все, по умолчанию 3 - удалить все старее 3 дней')
    args = parser.parse_args()
    log.info('arguments: %s', str(args))

    run(args.file_name, args.days_ago, args.remove_old_prices_days)
