# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json  # noqa: I
from datetime import datetime, time, timedelta
from functools import partial

from common.db.mds.clients import mds_s3_common_client
from common.models_abstract.schedule import ExpressType
from common.models_utils.i18n import RouteLTitle
from common.utils.date import MSK_TZ, timedelta_to_str_hours_and_minutes
from common.utils.lock import lock
from travel.rasp.library.python.common23.logging import log_run_time
from travel.rasp.library.python.common23.logging.scripts import script_context


class BaseSuburbansGenerator(object):
    """
    Базовый класс для генерации данных о ближайших электричках для ПП.
    """
    def __init__(self, log):
        self.script_name = None
        self.stations_file = None
        self.suburbans_file = None
        self.fn_settlements = {}
        self.log = log
        self.log_run_time = partial(log_run_time, logger=log)

    def run_script(self):
        """
        Генерация данных о ближайших электричках для ПП.
        Данные скалыдваются в mds.
        При запросе в export от ПП данные достаются из mds,
        и к ним добавляются данные об опозданиях.
        """
        with lock(self.script_name), script_context(self.script_name), self.log_run_time('Generating data for pp'):
            with self.log_run_time('Generating stations data'):
                self.generate_stations_data()
                stations = self.prepare_stations_data()
                self._write_to_mds(stations, self.stations_file)

            with self.log_run_time('Generating suburban_threads data'):
                suburbans = self.generate_suburban_threads_data()
                self._write_to_mds(suburbans, self.suburbans_file)

    def _write_to_mds(self, data, mds_key):
        data_dump = json.dumps(data, ensure_ascii=False, encoding='utf-8')
        with self.log_run_time('Save data to mds'):
            mds_s3_common_client.save_data(data=data_dump, key=mds_key)

    def prepare_stations_data(self):
        result = {}
        for fn_settlement in self.fn_settlements.values():
            result[str(fn_settlement.settlement._geo_id)] = fn_settlement.prepare_data()
        return result

    @staticmethod
    def filter_segments(segments, station_from, now_aware):
        """
        Фильтрация списка сегментов ближайших электричек
        Отбрасываются сегменты отстоящие более чем на сутки от первого сегмента
        Также оставляется не более 10 электричек, идущих позднее чем через 30 минут после текущего времени
        """
        local_today = now_aware.astimezone(station_from.pytz).date()
        end_limit = datetime.combine(local_today, time(0, 0)) + timedelta(2)
        end_limit = station_from.pytz.localize(end_limit)
        half_hour = timedelta(minutes=30)

        nearest_segments = []
        first_start = None
        extra_count = 0

        for segment in segments:
            if segment.departure <= now_aware:
                continue

            if segment.departure >= end_limit:
                break

            if first_start is not None and segment.departure > first_start + timedelta(hours=23, minutes=59):
                break

            if segment.departure >= now_aware + half_hour:
                extra_count += 1
                if extra_count > 10:
                    break

            if first_start is None:
                first_start = segment.departure

            nearest_segments.append(segment)

        return nearest_segments

    @staticmethod
    def prepare_segment_data(segment):
        """
        Подготовка данных по сегменту нитки для json
        """
        thread = segment.thread
        departure = segment.departure
        rtstation_from = segment.rtstation_from
        rtstation_to = segment.rtstation_to
        RouteLTitle.fetch([thread.L_title])

        thread_data = {
            'name': {
                'ru': thread.L_title(lang='ru', short=True),
                'uk': thread.L_title(lang='uk', short=True),
                'tr': thread.L_title(lang='tr', short=True),
            },
            'uid': thread.uid or u'',
        }
        if thread.express_type == ExpressType.EXPRESS:
            thread_data['express'] = 1

        t_subtype = thread.t_subtype
        if t_subtype:
            thread_data['transport_subtype'] = {
                'ru': t_subtype.L_title_short(lang='ru'),
                'uk': t_subtype.L_title_short(lang='uk'),
                'tr': t_subtype.L_title_short(lang='tr'),
            }

        thread_data['time'] = departure.isoformat().decode('utf8')

        thread_start_datetime = rtstation_from.calc_thread_start_date(
            event_tz=MSK_TZ, event='departure', event_date=departure.date())

        tz_arrival = rtstation_from.tz_arrival
        tz_departure = rtstation_from.tz_departure
        thread_data['from_id'] = rtstation_from.station.id
        thread_data['from_arrival'] = None if tz_arrival is None else int(tz_arrival)
        thread_data['from_departure'] = None if tz_departure is None else int(tz_departure)
        tz_thread_start_date = datetime.combine(thread_start_datetime, thread.tz_start_time).isoformat()
        thread_data['tz_thread_start_date'] = tz_thread_start_date.decode('utf8')

        # Электрички не могут идти больше суток, поэтому укладываемся в формат HH:MM
        thread_data['travel_time'] = timedelta_to_str_hours_and_minutes(segment.duration)

        thread_data['touch_url'] = (
            'https://t.rasp.yandex.ru/thread/{thread}/?'
            'departure={departure}&tt={tt}&station_from={st_from}&station_to={st_to}'.format(
                thread=thread.uid,
                departure=departure.date(),
                tt='suburban',
                st_from=rtstation_from.station.id,
                st_to=rtstation_to.station.id
            ))
        return thread_data

    def generate_stations_data(self):
        raise NotImplementedError

    def generate_suburban_threads_data(self):
        raise NotImplementedError
