# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from common.models.geo import ExternalDirection
from common.models.transport import TransportType
from route_search.models import ZNodeRoute2


def _prepare_station(station):
    """
    Подготовка данных о станции для json
    """
    return {
        'id': station.id,
        'lon': station.longitude,
        'lat': station.latitude,
        'name': {
            'ru': station.L_title(lang='ru'),
            'uk': station.L_title(lang='uk'),
            'tr': station.L_title(lang='tr'),
        }
    }


class ForNearestMainStation(object):
    """
    Станция для поиска главных станций городов
    """
    def __init__(self, station):
        self.station = station

    def prepare_data(self):
        return _prepare_station(self.station)


class ForNearestStation(object):
    """
    Станция для поиска ближайшей пары - центра пригородной зоны
    """
    def __init__(self, station, is_default, log):
        self.log = log
        self.station = station
        self.is_default = is_default
        self.base_station = None
        self.base_found = False
        self._find_base_station()

    def _find_base_station(self):
        try:
            zone_id = self.station.suburban_zone_id
            self.base_station = ExternalDirection.objects.filter(
                externaldirectionmarker__station=self.station, suburban_zone__id=zone_id
            )[0].base_station
        except IndexError:
            self.log.debug('Не нашли базовой станции пропускаем %s %s', self.station.id, self.station.title)
            return

        qs1 = ZNodeRoute2.objects.filter(station_from=self.station, station_to=self.base_station,
                                         thread__t_type__id=TransportType.SUBURBAN_ID)
        qs2 = ZNodeRoute2.objects.filter(station_from=self.base_station, station_to=self.station,
                                         thread__t_type__id=TransportType.SUBURBAN_ID)

        if qs1.exists() or qs2.exists():
            self.base_found = True
        else:
            self.log.debug('Нет сообщения от до центра %s %s пропускаем %s %s',
                           self.base_station.id, self.base_station.title, self.station.id, self.station.title)

    def prepare_data(self):
        return {
            "default": self.is_default,
            "st": _prepare_station(self.station),
            "cst": _prepare_station(self.base_station)
        }


class ForNearestMainSettlement(object):
    """
    Город, центр пригородной зоны. Используется для поиска главных станций городов
    """
    def __init__(self, settlement):
        self.settlement = settlement
        self.fn_stations = []

    def prepare_data(self):
        result = {}
        if len(self.fn_stations) > 0:
            result['main_stations'] = [m.prepare_data() for m in self.fn_stations]
        return result


class ForNearestSettlement(object):
    """
    Город - не центр пригородной зоны.
    Используется для для поиска ближайшего центра пригородной зоны
    """
    def __init__(self, settlement):
        self.settlement = settlement
        self.fn_stations = []

    def prepare_data(self):
        result = {}

        valid = False
        for fn_station in self.fn_stations:
            valid = valid or fn_station.base_found

        if valid:
            result['stations'] = [s.prepare_data() for s in self.fn_stations if s.base_found]

        return result


class ForNearestMainSegment(object):
    pass
