# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import timedelta, datetime

from common.models.geo import SuburbanZone, StationMajority
from common.models.transport import TransportType
from travel.rasp.library.python.common23.date import environment
from stationschedule import get_schedule_class, SUBURBAN
from stationschedule.utils import EVENT_DEPARTURE
from travel.rasp.tasks.nearest_suburbans.base_generator import BaseSuburbansGenerator
from travel.rasp.tasks.nearest_suburbans.data_classes import ForNearestMainSettlement, ForNearestMainStation, ForNearestMainSegment


class MainStationsSuburbansGenerator(BaseSuburbansGenerator):
    def __init__(self, log):
        BaseSuburbansGenerator.__init__(self, log)
        self.script_name = 'update_nearest_main_stations_suburbans'
        self.stations_file = 'suburban/export_nearest_suburban_main_stations.json'
        self.suburbans_file = 'suburban/nearest_suburban_main_stations_departures.json'

    SUBURBANS_COUNT_LIMIT = 10

    def generate_stations_data(self):
        """
        Формирование словаря городов - центров пригородных зон
        Для каждого города перечисляются все главные станции (с минимальным majority_id)
        https://st.yandex-team.ru/RASPFRONT-6481
        """
        zones = SuburbanZone.objects.all().prefetch_related('settlement')
        self.fn_settlements = {}

        for zone in zones:
            fn_settlement = ForNearestMainSettlement(zone.settlement)

            min_majority = StationMajority.MAX_ID
            for station in zone.settlement.station_set.filter(t_type_id=TransportType.TRAIN_ID):
                if station.majority.id < min_majority:
                    min_majority = station.majority.id
                    fn_settlement.fn_stations = []
                if station.majority.id == min_majority:
                    fn_settlement.fn_stations.append(ForNearestMainStation(station))

            if len(fn_settlement.fn_stations) > 0:
                self.fn_settlements[zone.settlement.id] = fn_settlement

    def generate_suburban_threads_data(self):
        """
        Формирование перечня электричек, выходящих из главных городов в ближайшее время
        Возвращает словарь главных станций центров пригородных зон
        Для каждой станции возвращается список электричек, отправляющихся в ближайшее время (не более 10)
        https://st.yandex-team.ru/RASPFRONT-6481
        """
        now_aware = environment.now_aware()

        segments_by_stations = {}
        for fn_settlement in self.fn_settlements.values():
            for fn_station in fn_settlement.fn_stations:
                station = fn_station.station
                segments_by_stations[str(station.id)] = self._get_station_schedule(station, now_aware)
        return segments_by_stations

    def _get_station_schedule(self, station, now_aware):
        now_date = station.get_local_datetime(now_aware).date()
        segments = []
        for schedule_date in [now_date, now_date + timedelta(1)]:
            segments.extend(self._get_day_station_segments(station, schedule_date))

        nearest_segments = self.filter_segments(segments, station, now_aware)

        now_naive = datetime.now()
        for segment in nearest_segments:
            segment.rtstation_to = segment.thread.rtstation_set.filter(tz_departure__isnull=True)[0]

            departure = segment.rtstation_from.get_departure_dt(now_naive)
            arrival = segment.rtstation_to.get_arrival_dt(now_naive, segment.rtstation_from.pytz)
            segment.duration = arrival - departure

        return [self.prepare_segment_data(segment) for segment in nearest_segments]

    @staticmethod
    def _get_day_station_segments(station, schedule_date):
        schedule_cls = get_schedule_class(station, schedule_type=SUBURBAN, t_type_code=SUBURBAN)
        schedule = schedule_cls(station,
                                event=EVENT_DEPARTURE,
                                requested_direction='all',
                                schedule_date=schedule_date)
        schedule.build(schedule_date=schedule_date)
        segments = []
        for route in schedule.schedule:
            segment = ForNearestMainSegment()
            segment.thread = route.thread
            segment.rtstation_from = route.rtstation
            segment.departure = route.departure_dt
            segments.append(segment)

        return segments
