# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from common.models.geo import Station, SuburbanZone
from common.models.transport import TransportType
from travel.rasp.library.python.common23.date import environment
from route_search.shortcuts import find
from travel.rasp.tasks.nearest_suburbans.base_generator import BaseSuburbansGenerator
from travel.rasp.tasks.nearest_suburbans.data_classes import ForNearestSettlement, ForNearestStation


class NearestSuburbansGenerator(BaseSuburbansGenerator):
    def __init__(self, only_default, log):
        BaseSuburbansGenerator.__init__(self, log)
        self.only_default = only_default
        all_string = '' if only_default else '_all'
        self.script_name = 'update' + all_string + '_nearest_suburbans'
        self.stations_file = 'suburban/export_nearest_suburban_stations.json'
        self.suburbans_file = 'suburban/nearest_suburban_to_from_center' + all_string + '.json'

    def generate_stations_data(self):
        """
        Возвращает словарь, где ключ - geo_id города, а значение -
        список словарей из "ближайшей" станции (st) и станции в городе (базовой) (cst).
        Для этого берутся станции в порядке их важности (majority).
        Для первой встречной станции из самых важных указывается default=true.
        Если у станции нет города, у города нет geo_id, у города нет зоны
        или нет сообщения между станцией и базовой станцией, или станция принадлежит главному городу пригородной зоны,
        то станция в список stations не добавляется
        https://st.yandex-team.ru/RASPEXPORT-273
        """
        stations_qs = (Station.objects.filter(t_type__id=TransportType.TRAIN_ID).
                       prefetch_related('settlement').order_by('majority__id'))
        stations = [station for station in stations_qs if self._can_use_station(station)]

        centers_ids = {zone.settlement.id for zone in SuburbanZone.objects.all()}

        for station in stations:
            settlement = station.settlement
            if settlement.id not in centers_ids:
                if settlement.id not in self.fn_settlements:
                    fn_settlement = ForNearestSettlement(settlement)
                    self.fn_settlements[settlement.id] = fn_settlement
                    default = True
                else:
                    fn_settlement = self.fn_settlements[settlement.id]
                    default = False
                fn_station = ForNearestStation(station, default, self.log)

                if fn_station.base_found and (not self.only_default or default):
                    fn_settlement.fn_stations.append(fn_station)

        self.log.info('Сгенерированы данные о ближайших станциях города из всех станций в базе')

    def _can_use_station(self, station):
        if not station.settlement:
            self.log.debug('Станция без города, пропускаем %s %s', station.id, station.L_title)
            return False

        if not station.suburban_zone_id:
            self.log.debug('Не нашли зоны станции, пропускаем %s %s', station.id, station.L_title)
            return False

        if not station.settlement._geo_id:
            self.log.debug('У города не указан geo_id, пропускаем %s %s', station.id, station.L_title)
            return False

        return True

    def generate_suburban_threads_data(self):
        """
        Для каждой пары: станция, базовая станция направления станции выполняется поиск ближайших ниток.
        Для пары выполняется два поиска - с сторону города (базовой станции) и в обратную сторону.
        https://st.yandex-team.ru/RASPEXPORT-273
        """
        now_aware = environment.now_aware()

        segments_by_stations = {}
        for fn_settlement in self.fn_settlements.values():
            for fn_station in fn_settlement.fn_stations:
                if not self.only_default or fn_station.is_default:
                    center = fn_station.base_station
                    station = fn_station.station

                    nearest_from_center = self._generate_nearest_from_to(center, station, now_aware)
                    nearest_to_center = self._generate_nearest_from_to(station, center, now_aware)
                    segments_by_stations[str(station.id)] = self._prepare_station_segments(nearest_to_center,
                                                                                           nearest_from_center)
        return segments_by_stations

    def _generate_nearest_from_to(self, station_from, station_to, now_aware):
        local_today = now_aware.astimezone(station_from.pytz).date()
        segments = find(station_from, station_to, local_today, 'suburban')
        return self.filter_segments(segments, station_from, now_aware)

    @classmethod
    def _prepare_station_segments(cls, nearest_to_center, nearest_from_center):
        return {
            'tc': [cls.prepare_segment_data(s) for s in nearest_to_center],
            'fc': [cls.prepare_segment_data(s) for s in nearest_from_center],
        }
