# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging  # noqa
from functools import partial

from django.conf import settings
from pymongo import ASCENDING
from pymongo.errors import OperationFailure

from common.data_api.dzv.instance import dzv
from common.data_api.dzv.platfrom_text_replacer import PlatformTextReplacer
from common.data_api.platforms.client import get_dynamic_platform_collection
from common.data_api.sendr.api import Campaign
from common.db.mongo.bulk_buffer import BulkBuffer
from common.models.geo import CodeSystem, StationCode
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from common.utils.lock import lock
from travel.rasp.library.python.common23.logging import log_run_time
from travel.rasp.library.python.common23.logging.scripts import script_context

log = logging.getLogger(__name__)
log_run_time = partial(log_run_time, logger=log)


define_setting('RASP_DYNAMIC_PLATFORMS_ERROR_EMAIL', {
    Configuration.PRODUCTION: 'rasp-dynamic-platforms-error@yandex-team.ru',
    Configuration.TESTING: 'rasp-dynamic-platforms-error-testing@yandex-team.ru'
}, default='rasp-dynamic-platforms-error-testing@yandex-team.ru')
define_setting('RASP_DYNAMIC_PLATFORMS_ERROR_CAMPAIGN', {
    Configuration.PRODUCTION: 'OWQHZL14-RQ02',
    Configuration.TESTING: 'C3LM4N14-9941'
}, default='C3LM4N14-9941')


class PlatformsUpdater:
    def __init__(self, coll):
        self.replacer = PlatformTextReplacer()
        self.coll = coll

    def update(self, field_name, source, station):
        unparsed_track_numbers = []

        with BulkBuffer(self.coll, max_buffer_size=5000) as buff:
            for event in source:
                if not event['track_number'] or event['track_number'] == '0':
                    continue
                platform = self.replacer.get_platform_text(event['track_number'], station)
                if not platform:  # if regexp for station is not defined (replacer)
                    log.error('Unable to parse track number. Station: {}. Text: {}'.format(
                        station.id, event['track_number']))
                    unparsed_track_numbers.append(event['track_number'])
                    continue
                else:
                    buff.update_one(
                        {
                            'date': event['event_dt'].date().isoformat(),  # date of local station time
                            'station_id': station.id,
                            'train_number': event['train_number'],
                        },
                        {'$set': {
                            field_name: platform,
                        }},
                        upsert=True,
                    )
        return unparsed_track_numbers


def send_unparsed_track_numbers(unparsed_track_numbers):
    errors = []
    for station, track_number in unparsed_track_numbers:
        errors.append({
            'station_id': station.id,
            'station_title': station.title,
            'text': track_number
        })
    campaign_send = Campaign.create_rasp_campaign(settings.RASP_DYNAMIC_PLATFORMS_ERROR_CAMPAIGN)
    try:
        campaign_send.send(
            to_email=settings.RASP_DYNAMIC_PLATFORMS_ERROR_EMAIL,
            args={'errors': errors}
        )
    except Exception:
        log.exception("Can't send a letter to {}: {}".format(
            settings.RASP_DYNAMIC_PLATFORMS_ERROR_EMAIL,
            json.dumps(errors)
        ))


def update_dynamic_platforms():
    code_system = CodeSystem.objects.get(code='dzv')
    station_codes = list(StationCode.objects.filter(system=code_system).select_related('station'))
    coll = get_dynamic_platform_collection()

    # Date is the first in the index because of clearing queries by date
    coll.create_index(
        [('date', ASCENDING), ('station_id', ASCENDING), ('train_number', ASCENDING)],
        background=True
    )

    updater = PlatformsUpdater(coll)
    error_stations = []
    with log_run_time('updating platforms for {} station codes'.format(len(station_codes))):
        for station_code in station_codes:
            code = station_code.code
            if not code.isdigit():  # not integer strings may be if we need to switch off updating for this station
                continue

            dzv_data = dzv.get_platforms(int(code))
            departure_unparsed = updater.update('departure_platform', dzv_data['departure'], station_code.station)
            arrival_unaprsed = updater.update('arrival_platform', dzv_data['arrival'], station_code.station)

            for track_number in departure_unparsed + arrival_unaprsed:
                error_stations.append((station_code.station, track_number))

    if error_stations:
        send_unparsed_track_numbers(error_stations)


def run():
    fault_message = None
    try:
        with log_run_time('update_dynamic_platforms', logger=log):
            update_dynamic_platforms()
    except OperationFailure as ex:
        fault_message = 'update_dynamic_platforms failed: {};{}'.format(ex.code, ex.details)
        raise
    except Exception as ex:
        fault_message = 'update_dynamic_platforms failed: {}'.format(ex.message)
        raise
    finally:
        if fault_message:
            log.exception(fault_message)


def lock_aware_run():
    with lock('lock_update_dynamic_platforms', database_name=settings.DYNAMIC_PLATFORMS_DB_ALIAS), \
         script_context('update_dynamic_platforms'):
        run()


if __name__ == '__main__':
    lock_aware_run()
