# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from django import forms
from django.core.exceptions import ObjectDoesNotExist, NON_FIELD_ERRORS, ValidationError

from common.utils.date import fromSQLDate, fromSQLdatetime
from common.models.geo import Station, Point
from common.models.transport import TransportType
from common.xgettext.i18n import dynamic_gettext, gettext
from geosearch.views.point import StopWordError, TooShortError, NoPointsError, StationPointSearch
import settings


class BaseForm(forms.Form):
    fields_order_error = []

    def __init__(self, *args, **kwargs):
        self.request = kwargs.pop('request')

        super(BaseForm, self).__init__(*args, **kwargs)

    def add_error(self, field, error):
        if field not in self.errors:
            if field != NON_FIELD_ERRORS and field not in self.fields:
                raise ValueError("'%s' has no field named '%s'." % (self.__class__.__name__, field))

            self.errors[field] = self.error_class()

        self.errors[field].append(error)

        if field in self.cleaned_data:
            del self.cleaned_data[field]

    def list_errors(self):
        errors = []

        for field_name in self.fields_order_error:
            if field_name not in self.errors:
                continue

            error = map(dynamic_gettext, self.errors[field_name])

            if error not in errors:
                errors.extend([error, '<br/>'])

        return errors


class ThreadForm(BaseForm):
    departure = forms.CharField(required=False)
    departure_from = forms.CharField(required=False)
    station_from = forms.IntegerField(required=False)
    station_to = forms.IntegerField(required=False)
    point_from = forms.CharField(required=False)
    point_to = forms.CharField(required=False)

    def clean_departure(self):
        cleaned_departure = self.cleaned_data['departure']

        if not cleaned_departure:
            return

        try:
            return fromSQLDate(cleaned_departure)

        except (ValueError, TypeError):
            pass

    def clean_departure_from(self):
        if 'departure_from' in self.cleaned_data and self.cleaned_data['departure_from']:
            try:
                return fromSQLdatetime(self.cleaned_data['departure_from'])
            except (ValueError, TypeError):
                pass
        return None

    def clean_station_from(self):
        sid = self.cleaned_data['station_from']

        if not sid:
            return

        try:
            return Station.objects.select_related('settlement').get(pk=sid)

        except (Station.DoesNotExist, ValueError):
            pass

    def clean_station_to(self):
        sid = self.cleaned_data['station_to']

        if not sid:
            return

        try:
            return Station.objects.select_related('settlement').get(pk=sid)

        except (Station.DoesNotExist, ValueError):
            pass

    def clean_point_from(self):
        point_key = self.cleaned_data['point_from']

        if not point_key:
            return

        try:
            return Point.get_by_key(point_key)

        except (ObjectDoesNotExist, ValueError):
            pass

    def clean_point_to(self):
        point_key = self.cleaned_data['point_to']

        if not point_key:
            return

        try:
            return Point.get_by_key(point_key)

        except (ObjectDoesNotExist, ValueError):
            pass


class StationSearchForm(forms.Form):
    pointName = forms.CharField(max_length=255, error_messages={'required': 'Введите название станции или города'})
    pointId = forms.CharField(max_length=255, required=False)

    current = forms.BooleanField(required=False)
    other = forms.BooleanField(required=False)

    fields_order_error = ('pointName',)

    def __init__(self, *args, **kwargs):
        self.country_id = kwargs.pop('country_id', settings.OUR_COUNTRIES[0])

        super(StationSearchForm, self).__init__(*args, **kwargs)

    def get_filter(self):
        if not self.is_valid():
            return

        data = self.cleaned_data

        if data['current'] and not data['other']:
            return 'current'

        if data['other'] and not data['current']:
            return 'other'

    def clean(self):
        if 'pointName' not in self.cleaned_data:
            raise ValidationError(gettext(
                u"Введите название станции или города."
            ))

        try:
            points = StationPointSearch.find_point(
                self.cleaned_data['pointName'],
                self.cleaned_data['pointId'],
                country_id=self.country_id,
                country_filter_type=self.get_filter()
            )
        except StopWordError:
            raise ValidationError(gettext(
                u"Введенное вами название слишком общее.<br/>"
                u"Пожалуйста, уточните его."
            ))
        except TooShortError:
            raise ValidationError(gettext(
                u"Введенное вами название слишком короткое.<br/>"
                u"Пожалуйста, введите более длинное название (от 3 символов)."
            ))
        except NoPointsError:
            raise ValidationError(gettext(
                u"К сожалению, у нас нет информации о станции с таким "
                u"названием."
            ))

        self.cleaned_data['points'] = points

        return self.cleaned_data


class RouteSearchForm(BaseForm):
    number = forms.CharField(max_length=255, error_messages={'required': 'Введите номер рейса или поезда.'})

    plane = forms.BooleanField(required=False)
    train = forms.BooleanField(required=False)
    suburban = forms.BooleanField(required=False)
    bus = forms.BooleanField(required=False)

    fields_order_error = ('number',)

    def get_t_types_ids(self):
        codes = [code for code in ('plane', 'train', 'suburban', 'bus') if self.cleaned_data.get(code)]

        return [t_type.id for t_type in TransportType.objects.all_cached() if t_type.code in codes]
