# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from datetime import timedelta

from django import forms
from django.conf import settings

from common.xgettext.i18n import gettext, mark_gettext
from common.utils.date import uni_strftime
from common.utils.dateparser import parse_human_date
from common.models.geo import TimeZoneMixin, Settlement, Station
from geosearch.views.point import (PointSearch, SuburbanPointSearch, NoPointsError, InvalidPointKey,
                                   StopWordError, TooShortError, SuggestsPointSearch)
from geosearch.views.pointtopoint import SamePointError, process_points_lists

from travel.rasp.touch.forms import BaseForm


class SearchForm(BaseForm):
    fromName = forms.CharField(label='Откуда', max_length=100, required=False)
    fromId = forms.CharField(required=False)
    toName = forms.CharField(label='Куда', max_length=100, required=False)
    toId = forms.CharField(required=False)
    when = forms.CharField(required=False, max_length=30, label='Когда')
    express = forms.BooleanField(required=False)
    aeroex = forms.BooleanField(required=False)
    validationOff = forms.CharField(required=False, max_length=5)

    fields_order_error = ('fromName', 'toName', 'when')

    def get_searcher(self, suggests=False):
        return SuggestsPointSearch if suggests else self.searcher

    def __init__(self, *args, **kwargs):
        self.ttype = kwargs.pop('ttype', None)

        if self.ttype == 'suburban':
            self.searcher = SuburbanPointSearch
            self.suggests_type = 'suburban'

        else:
            self.searcher = PointSearch
            self.suggests_type = 'common'

        super(SearchForm, self).__init__(*args, **kwargs)

        self.today = self.request.now.date()

        self.now = self.request.now.replace(tzinfo=None)
        self.loc_today = self.request.client_city.get_local_datetime(self.now).date()

    def get_when(self):
        when = self.data['when'] if 'when' in self.data else self.loc_today

        if when.year != self.loc_today.year:
            return uni_strftime('%d %B %Y', when)

        return uni_strftime('%d %B', when)

    def get_delta(self):
        when = self.data['when'] if 'when' in self.data else self.loc_today

        return (when - self.loc_today).days

    def is_now_today(self):
        return 'when' not in self.data or self.loc_today == self.data['when']

    def _clean_when(self, today=None):
        cleaned_when = self.cleaned_data.get('when')

        if not today:
            today = self.loc_today

        if cleaned_when:
            try:
                d = parse_human_date(cleaned_when, today, timedelta(settings.DAYS_TO_PAST))

            except ValueError:
                self.add_error('when', mark_gettext('Неизвестный формат даты «туда».<br/>Укажите дату в формате «25 января».'))
                self.data['when'] = today

                return

        else:
            d = today

        if not d:
            self.add_error('when', mark_gettext(('В версии Яндекс.Расписаний для тачевых устройств отсутствует возможность '
                                                'поиска "на все дни".<br/>Попробуйте указать конкретную дату')))
            self.data['when'] = today  # Для рендеринга формы на странице с ошибками

            return

        self.data['when'] = d

        return d

    def get_search_params(self, is_raw=False):
        origin = self.data

        if self.is_valid() and not is_raw:
            origin = self.cleaned_data

        return {
            'fromId': origin.get('fromId', ''),
            'fromName': origin.get('fromName', ''),
            'toId': origin.get('toId', ''),
            'toName': origin.get('toName', ''),
            'when': '' if self.is_now_today() else self.get_when(),
            'delta': self.get_delta()
        }

    def clean(self):
        from_point_list = to_point_list = None

        suggest_field = self['validationOff'].data
        suggest_from = suggest_to = False

        if suggest_field == 'from':
            suggest_from = True
        elif suggest_field == 'to':
            suggest_to = True

        client_city = self.request and self.request.client_city or None
        national_version = self.request and self.request.NATIONAL_VERSION or None

        try:
            from_point_list = (
                self.get_searcher(suggests=suggest_from).find_point(
                    self.cleaned_data.get('fromName'),
                    t_type=self.ttype,
                    point_key=self.cleaned_data.get('fromId'),
                )
            )
        except NoPointsError:
            self.add_error('fromName', mark_gettext('''К сожалению, у нас нет информации о пункте отправления.<br/>
                                        Пожалуйста, проверьте правильность написания или выберите другой город.'''))

        except StopWordError:
            self.add_error('fromName', mark_gettext(('Введенное вами название пункта отправления слишком общее.<br/>'
                                                    'Пожалуйста, уточните его.')))

        except InvalidPointKey:
            self.add_error('fromName', mark_gettext('Неизвестный пункт отправления'))

        except TooShortError:
            self.add_error('fromName', mark_gettext('Слишком короткое название для пункта отправления'))

        except Station.DoesNotExist:
            self.add_error('fromName', mark_gettext('Неизвестная станция отправления'))

        except Settlement.DoesNotExist:
            self.add_error('fromName', mark_gettext('Неизвестный город отправления'))

        try:
            to_point_list = (
                self.get_searcher(suggests=suggest_to).find_point(
                    self.cleaned_data.get('toName'),
                    t_type=self.ttype,
                    point_key=self.cleaned_data.get('toId'),
                )
            )
        except NoPointsError:
            message = mark_gettext('''К сожалению, у нас нет информации о пункте прибытия.<br/>
                                    Пожалуйста, проверьте правильность написания или выберите другой город.''')
            self.add_error('toName', message)

        except StopWordError:
            self.add_error('toName', mark_gettext(('Введенное вами название пункта прибытия слишком общее.'
                                                   '<br/> Пожалуйста, уточните его.')))

        except InvalidPointKey:
            self.add_error('toName', mark_gettext('Неизвестный пункт прибытия'))

        except TooShortError:
            self.add_error('toName', mark_gettext('Слишком короткое название для пункта прибытия'))

        except Station.DoesNotExist:
            self.add_error('toName', mark_gettext('Неизвестная станция прибытия'))

        except Settlement.DoesNotExist:
            self.add_error('toName', mark_gettext('Неизвестный город прибытия'))

        if from_point_list is None or to_point_list is None:
            self._clean_when()

            return self.cleaned_data

        try:
            from_point_list, to_point_list = process_points_lists(
                from_point_list,
                to_point_list,
                client_city=client_city,
                suburban=self.ttype == 'suburban',
                disable_reduce_from=suggest_from,
                disable_reduce_to=suggest_to
            )
        except SamePointError:
                self.add_error('fromName', mark_gettext('Укажите различные пункты отправления и прибытия'))
                self.add_error('toName', mark_gettext('Укажите различные пункты отправления и прибытия'))
                self._clean_when()

                return self.cleaned_data

        self.cleaned_data['ambiguous'] = False

        if from_point_list.has_variants() or to_point_list.has_variants():
            from_point_list.process()
            to_point_list.process()

            if not from_point_list.dont_rearrange:
                from_point_list.rearrange_variants(
                    client_city, self.data['fromName'], national_version)

            if not to_point_list.dont_rearrange:
                to_point_list.rearrange_variants(
                    client_city, self.data['toName'], national_version)

            self.cleaned_data['ambiguous'] = True

        else:
            self.cleaned_data['fromName'] = self.data['fromName'] = from_point_list.point.get_popular_title()
            self.cleaned_data['toName'] = self.data['toName'] = to_point_list.point.get_popular_title()

        self.cleaned_data['from'] = from_point_list
        self.cleaned_data['to'] = to_point_list

        from_point = from_point_list.point

        if isinstance(from_point, TimeZoneMixin):
            self._clean_when(from_point.get_local_datetime(self.now).date())
        else:
            self._clean_when()

        return self.cleaned_data


class DirectionSearchForm(SearchForm):
    fromName = forms.CharField(label='Откуда', max_length=100, required=False)
    fromId = forms.CharField(required=False)
    toName = forms.CharField(label='Куда', max_length=100, required=False)
    toId = forms.CharField(required=False)
    when = forms.CharField(required=False, max_length=30, label='Когда')

    validationOff = forms.CharField(required=False, max_length=5)

    direction = forms.CharField(label='Направление', required=False)
    all_days = forms.BooleanField(required=False)
    mode = forms.CharField(max_length=255, required=False)

    def __init__(self, *args, **kwargs):
        kwargs['ttype'] = 'suburban'

        super(DirectionSearchForm, self).__init__(*args, **kwargs)

    def get_when(self):
        if self['all_days'].data:
            return gettext('на все дни')

        return super(DirectionSearchForm, self).get_when()

    def get_when_value(self):
        if self['all_days'].data:
            return uni_strftime('%d %B', self.loc_today)

        return self.get_when()

    def get_delta(self):
        if self['all_days'].data:
            return 'all'

        return super(DirectionSearchForm, self).get_delta()

    def get_search_params(self, is_raw=False, city_id=None, direction=None):
        params = super(DirectionSearchForm, self).get_search_params(is_raw=is_raw)

        if self['all_days'].data:
            params['all_days'] = '1'

        params['city'] = city_id

        if direction:
            params['direction'] = {
                'name': direction.title,
                'code': direction.code
            }

        return params

    def _clean_when(self, today=None):
        cleaned_when = self.cleaned_data.get('when')
        all_days = self.cleaned_data.get('all_days')

        if all_days:  # when игнорим
            cleaned_when = ''

        if not today:
            today = self.loc_today

        if cleaned_when:
            try:
                d = parse_human_date(cleaned_when, today, timedelta(settings.DAYS_TO_PAST))

                # есть ветки в логике parse_human_date, когда возвращается None
                if not d:
                    raise ValueError(u'Unknown date format %r' % cleaned_when)

            except ValueError:
                self.add_error('when', mark_gettext('Неизвестный формат даты «туда».<br/>Укажите дату в формате «25 января».'))
                self.data['when'] = today

                return

        else:
            d = today

        self.data['when'] = d

        return d

    def clean(self):
        result = super(DirectionSearchForm, self).clean()

        if self['all_days'].data:
            result['all_days'] = '1'

        mode = 'day'

        if self['all_days'].data:
            mode = 'all_days'
        else:
            if self.is_now_today():
                if self['mode'].data != 'day':
                    mode = 'nearest'
                else:
                    mode = 'today'

        result['mode'] = mode

        return result
