(function ()
{
    var Url = BEM.blocks['i-modern-url'];
    var history = window.history;
    var hasPushState = Boolean(history && history.pushState);
    var hasBack = Boolean(history && history.back);

    /**
     * Класс для работы c текущим URL страницы. Синхронизирует изменения URL с историей браузера.
     */
    var Location = $.inherit($.observable, {
        __constructor: function () {
            // Получаем текущий URL.
            this._url = Url.parse(window.location.pathname + window.location.search);

            this._url.on('change', function () {
                this.trigger('change');
            }, this);

            // в ФФ событие бросается даже когда hash не изменился
            var oldHash = window.location.hash;
            $(window).on('hashchange', function () {
                if (window.location.hash !== oldHash) {
                    this.trigger('hashchange');
                }
                oldHash = window.location.hash;
            }.bind(this));

            if (hasPushState) {
                var that = this;
                $(window).on('popstate', function () {
                    var newUrl = Url.parse(window.location.pathname + window.location.search);
                    that._url.setPath(newUrl.getPath());
                    that._url.replaceParams(newUrl.getParams());
                });
            }
        },

        getPath: function () {
            return this._url.getPath();
        },

        /**
         * Возвращает параметры текущего URL страницы.
         *
         * @returns {Object}
         */
        getParams: function () {
            return this._url.getParams();
        },

        /**
         * Изменяет параметры текущего URL без создания записи в истории браузера.
         *
         * @param {Object} params Новые параметры.
         */
        rewrite: function (params) {
            this._url.replaceParams(params);
            if (hasPushState) {
                history.replaceState(null, document.title, this._url.render());
            }
        },

        /**
         * Изменяет параметры текущего URL и создает запись в истории браузера.
         *
         * @param {String} path - Новый путь (необязательный параметр)
         * @param {Object} params Новые параметры.
         * @param {Boolean} [extend=false] Если `true`, то текущие параметры будут расширены.
         */
        navigate: function (path, params, extend) {
            var silent = true;
            var oldUrl = this._url.render();

            // Необязательность первого аргумента
            if (arguments.length < 3 && typeof path === 'object') {
                extend = params;
                params = path;
                path = null;
            }

            var pathChanged = path && this._url.setPath(path, silent);
            var paramsChanged = false;

            if (extend) {
                paramsChanged = this._url.setParams(params, silent);
            } else {
                paramsChanged = this._url.replaceParams(params, silent);
            }

            var newUrl = this._url.render();

            // убедимся что урл действительно изменился
            if (oldUrl === newUrl) {
                return;
            }

            if (!hasPushState) {
                window.location.href = newUrl;
                return;
            }

            history.pushState(null, document.title, newUrl);

            if (pathChanged || paramsChanged) {
                this.trigger('change');
            }
        },

        /**
         * Изменяет текущий URL и создает запись в истории браузера.
         *
         * @param {String} href
         * @param {Boolean} [extend=false] Если `true`, то текущие параметры будут расширены.
         */
        navigateHref: function (href, extend) {
            var url = Url.parse(href);

            this.navigate(url.getPath(), url.getParams(), extend);
        },

        getRelativeUrl: function () {
            return window.location.pathname + window.location.search;
        },

        back: function (newUrl) {
            if (hasBack) {
                // iPhone 6 некидает popstate при history.back => нужно менять урл ругами
                this._url.setPath(newUrl.getPath());
                this._url.replaceParams(newUrl.getParams());
                history.back();
            }
        },
        getHash: function () {
            var hash = Url.getHash();

            return hash.split('&').reduce(function (result, pair) {
                var eq = pair.indexOf('=');
                var name;
                var value;

                if (eq >= 0) {
                    name = pair.substr(0, eq);
                    value = pair.substr(eq + 1);
                } else {
                    name = pair;
                    value = '';
                }

                if (name) {
                    result[name] = value;
                }

                return result;
            }, {});
        },

        setHash: function (newHash) {
            var fragments = Object.keys(newHash).map(function (key) {
                return key + '=' + newHash[key];
            });

            var oldHash = window.location.hash;

            // Some browsers require that `hash` contains a leading #.
            window.location.hash = '#' + fragments.join('&');

            return oldHash !== window.location.hash;
        },

        hasPushState: function () {
            return hasPushState;
        },

        /**
         * Использовать только для чтения!
         * @returns {Url|*|Location._url}
         */
        getUrl: function () {
            return this._url;
        }
    });

    BEM.decl('i-location', {}, new Location());

}());
