/**
 * Created by mangin on 10/08/15.
 */
(function () {
    var hasOwnProperty = Object.prototype.hasOwnProperty,
        querystring = BEM.blocks['i-querystring'];

    /**
     * Класс для работы с URL.
     */
    var Url = $.inherit($.observable, {
        __constructor: function () {
            this._path = '';
            this._params = {};
        },

        /**
         * Возвращает путь.
         *
         * @returns {String}
         */
        getPath: function () {
            return this._path;
        },

        /**
         * Задает путь.
         *
         * @param {String} path
         * @param {Boolean} silent
         */
        setPath: function (path, silent) {
            if (this._path !== path) {
                this._path = path;

                if (!silent) {
                    this.trigger('change');
                }

                return true;
            }
        },

        /**
         * Возвращает значение параметра.
         *
         * @param {String|undefined} name
         */
        getParam: function (name) {
            return this._params[name];
        },

        /**
         * Устанавливает значения параметра.
         *
         * @param {String} name
         * @param {String} value
         */
        setParam: function (name, value) {
            if (this._params[name] !== value) {
                this._params[name] = value;
                this.trigger('change');
                return true;
            }
        },

        /**
         * Удаляет параметр по имени.
         *
         * @param {String} name
         */
        removeParam: function (name) {
            if (hasOwnProperty.call(this._params, name)) {
                delete this._params[name];
                this.trigger('change');
                return true;
            }
        },

        /**
         * Возвращает копию параметров.
         *
         * @returns {Object}
         */
        getParams: function () {
            return $.extend({}, this._params);
        },

        /**
         * Устанавливает несколько параметров.
         *
         * @param {Object} newParams
         * @param {Boolean} [silent]
         */
        setParams: function (newParams, silent) {
            var hasChanges = false;
            var params = this._params;
            Object.keys(newParams).forEach(function (name) {
                if (params[name] !== newParams[name]) {
                    params[name] = newParams[name];

                    if (params[name] === null) {
                        delete params[name];
                    }

                    hasChanges = true;
                }
            });
            if (hasChanges) {
                if (!silent) {
                    this.trigger('change');
                }
            }

            return this;
        },

        /**
         * Заменяет существующие параметры новыми.
         *
         * @param {Object} newParams
         * @param {Boolean} silent
         */
        replaceParams: function (newParams, silent) {
            var hasChanges = false;
            var params = this._params;

            Object.keys(params).forEach(function (name) {
                if (!hasOwnProperty.call(newParams, name)) {
                    hasChanges = true;
                    delete params[name];
                }
            });

            Object.keys(newParams).forEach(function (name) {
                if (params[name] !== newParams[name]) {
                    params[name] = newParams[name];
                    hasChanges = true;
                }
            });

            if (hasChanges) {
                if (!silent) {
                    this.trigger('change');
                }
                return true;
            }
        },

        /**
         * Возвращает строковое представление урла.
         *
         * @returns {String}
         */
        render: function () {
            var qs = querystring.stringify(this._params);
            return this._path + (qs ? '?' + qs : '');
        }
    }, {
        /**
         * Разбирает урл и возвращает объекта класса Url.
         *
         * @static
         * @param {String} urlStr
         * @returns {Url}
         */
        parse: function (urlStr) {
            var url = new Url();
            var urlParts = urlStr.split('?');
            url.setPath(urlParts[0]);
            if (urlParts.length === 2) {
                urlParts = urlParts[1].split('#'); // обрезаем хэш
                url.setParams(querystring.parse(urlParts[0]));
            }
            return url;
        },

        getHash: function () {
            var match = window.location.href.match(/#(.*)$/);
            return match ? decodeURIComponent(match[1]) : '';
        }
    });

    BEM.decl('i-modern-url', {}, Url);
}());
