# -*- coding: utf-8 -*-

import json
from datetime import datetime, timedelta

from django import forms
from django.conf import settings
from django.core.exceptions import ObjectDoesNotExist
from django.forms import ValidationError
from django.forms.forms import BoundField
from django.http import QueryDict
from django.utils.safestring import mark_safe

from common.dynamic_settings.default import conf
from common.models.currency import Currency
from common.models.geo import Point, Station, Settlement
from common.models.schedule import RThread, RTStation, RThreadType
from travel.rasp.library.python.common23.date import environment
from common.utils.date import fromSQLDate
from common.utils.locations import composeurl
from common.utils.order_data import check_order_data_digest
from common.views.tariffs import DisplayInfo
from route_search.models import RThreadSegment


class AsHidden(object):
    def as_hidden(self):
        """
        Returns this form rendered entirely as hidden fields.
        """
        output = []

        for name, field in self.fields.items():
            if not name in self.skipped_fields:
                bf = BoundField(self, field, name)

                output.append(bf.as_hidden())

        return mark_safe(u'\n'.join(output))


class OrderForm(forms.Form, AsHidden):
    choice = forms.CharField(required=False)
    n_adult = forms.IntegerField(required=False)
    n_child = forms.IntegerField(required=False)
    n_infant = forms.IntegerField(required=False)
    discount = forms.BooleanField(required=False)
    point_from = forms.CharField(required=False)
    point_to = forms.CharField(required=False)
    seats = forms.CharField(required=False)
    et = forms.CharField(required=False)

    skipped_fields = ['choice', 'discount']

    def clean_choice(self):
        choice = self.cleaned_data['choice']

        if choice:
            order_data = QueryDict(choice.encode('utf-8')).dict()

            if order_data and not check_order_data_digest(order_data):
                raise forms.ValidationError('Invalid MAC')

            return order_data

    def clean_n_adult(self):
        if self.cleaned_data['n_adult'] is None:
            return 1

        return self.cleaned_data['n_adult']

    def clean_point_from(self):
        if self.cleaned_data['point_from']:
            try:
                return Point.get_by_key(self.cleaned_data['point_from'])
            except (ValueError, ObjectDoesNotExist):
                raise forms.ValidationError('')

    def clean_point_to(self):
        if self.cleaned_data['point_to']:
            try:
                return Point.get_by_key(self.cleaned_data['point_to'])
            except (ValueError, ObjectDoesNotExist):
                raise forms.ValidationError('')

    def clean_seats(self):
        data = self.cleaned_data['seats']

        if data:
            try:
                return json.loads(data)
            except ValueError:
                pass

            return [{"number": number} for number in data.split(',')]

    def clean(self):
        super(OrderForm, self).clean()

        if self.cleaned_data.get('choice') and 'tariff' in self.cleaned_data['choice']:

            choice = self.cleaned_data['choice']

            data = choice['tariff'].split(' ')

            tariff = float(data[0])

            currency = data[1] if len(data) == 2 else Currency.BASE_CURRENCY

            self.cleaned_data['category'] = choice.get('cls')
            self.cleaned_data['tariff'] = tariff
            self.cleaned_data['currency'] = currency
            self.cleaned_data['total'] = tariff * self.cleaned_data['n_adult']

        return self.cleaned_data


class SegmentForm(forms.Form):
    station_from = forms.IntegerField()
    station_to = forms.IntegerField()
    thread = forms.CharField(max_length=50)
    date = forms.CharField(max_length=20)
    time_zone = forms.CharField(max_length=10, required=False)

    def clean_station_from(self):
        try:
            return Station.objects.get(id=self.cleaned_data['station_from'])
        except Station.DoesNotExist:
            raise ValidationError(u'Неизвестная станция отправления')

    def clean_station_to(self):
        try:
            return Station.objects.get(id=self.cleaned_data['station_to'])
        except Station.DoesNotExist:
            raise ValidationError(u'Неизвестная станция прибытия')

    def clean_thread(self):
        try:
            return RThread.objects.get(uid=self.cleaned_data['thread'])
        except RThread.DoesNotExist:
            raise ValidationError(u'Неизвестный маршрут')

    def clean_date(self):
        try:
            return fromSQLDate(self.cleaned_data['date']).date()
        except ValueError:
            raise ValidationError(u'Некорректный формат даты отправления.')

    def clean(self):
        if len(self.errors) > 0:
            return self.cleaned_data

        # Рейс туда
        segment = None

        # Дни хождения рейса по заданным станциям
        departure_times = []

        # Перебираем все нитки
        thread = self.cleaned_data['thread']  # Выбранная

        if thread.t_type.code != 'train':
            raise ValidationError(u'Купить билет на этот тип транспорта невозможно')

        # Лимит по выборке дней хождения
        if thread.t_type.code == 'plane':
            days_limit = settings.AGENT_LIMIT_DAYS_FORWARD

        elif thread.t_type.code == 'bus':
            days_limit = settings.BUS_LIMIT_DAYS_FORWARD

        else:
            days_limit = conf.TRAIN_ORDER_DEFAULT_DEPTH_OF_SALES

        self.cleaned_data['days_limit'] = days_limit

        last_departure_day = environment.today() + timedelta(days=days_limit - 1)

        try:
            date_tz = Settlement.hidden_manager.get(pk=int(self.cleaned_data['time_zone']))

        except (Settlement.DoesNotExist, ValueError, TypeError):
            date_tz = None

        # Выбранная нитка должна идти первой
        route_all_threads = [thread] + list(thread.route.rthread_set.filter(
            type__id=RThreadType.BASIC_ID).exclude(id=thread.id))

        for rthread in route_all_threads:
            s = RThreadSegmentSimple(rthread,
                                     self.cleaned_data['station_from'],
                                     self.cleaned_data['station_to'],
                                     from_date=self.cleaned_data['date'],
                                     from_date_tz=date_tz)

            # Для календаря нужны все нитки с правильным направлением
            if s.direction_is_valid:
                path = list(rthread.path)

                try:
                    rts_from = next((x for x in path if x.station == s.station_from))

                except StopIteration:
                    if s.station_from.settlement is None:
                        continue

                    try:
                        rts_from = next((x for x in path
                                         if x.station.settlement == s.station_from.settlement))

                    except StopIteration:
                        continue

                # добавляем дни хождения
                for day in rthread.get_run_date_list():
                    naive_start_dt = datetime.combine(day, rthread.tz_start_time)

                    departure_dt = rts_from.get_departure_dt(naive_start_dt, out_tz=s.station_from.pytz)

                    # отсеиваем ушедшие
                    if departure_dt < environment.now_aware():
                        continue

                    # Не считаем дальше предела
                    if departure_dt.date() > last_departure_day:
                        break

                    departure_times.append(departure_dt)

            if not segment and s.is_valid():
                # Выбираем сегмент
                segment = s
                self.cleaned_data['thread'] = rthread  # на всякий случай

        # Не нашли тред, неправильная ссылка
        if not segment:
            raise ValidationError(u'Маршрут не найден')

        # Мы не можем продавать билеты если поезд очень далеко или поезд ушел
        if all(x.date() != self.cleaned_data['date'] for x in departure_times):
            raise ValidationError(u'На выбранную дату нельзя купить билет')

        self.cleaned_data['segment'] = segment

        self.cleaned_data['departure_times'] = departure_times

        return self.cleaned_data


class OrderSegmentForm(OrderForm, SegmentForm):
    pass


class RThreadSegmentSimple(RThreadSegment):
    def __init__(self, thread=None, station_from=None, station_to=None,
                 from_date=None, from_date_tz=None,
                 rtstation_from=None, rtstation_to=None):
        """
        :param thread: нитка
        :param station_from: станция отправления
        :param station_to: станция прибытия
        :param from_date: дата отправления со station_from
        :param from_date_tz: временная зона для даты отправления
        """

        self.display_info = DisplayInfo()

        self.thread = thread
        self.station_from = station_from
        self.station_to = station_to

        if rtstation_from:
            self._rtstation_from = rtstation_from
        if rtstation_to:
            self._rtstation_to = rtstation_to

        rtstation_from = self.rtstation_from
        rtstation_to = self.rtstation_to

        if rtstation_from is None or rtstation_to is None:
            self.reset()

            return

        # Станции отправления и прибытия есть в маршруте этой нитки
        self.direction_is_valid = True

        thread_start_date = rtstation_from.calc_thread_start_date(
            event='departure', event_date=from_date, event_tz=from_date_tz)

        naive_start_dt = datetime.combine(thread_start_date, thread.tz_start_time)

        self.departure = rtstation_from.get_departure_dt(naive_start_dt, station_from.pytz)
        self.arrival = rtstation_to.get_arrival_dt(naive_start_dt, station_to.pytz)

        self.duration = self.arrival - self.departure

        self.start_date = thread_start_date

        self.now = environment.now_aware()
        self._init_data()

    @property
    def rtstation_from(self):
        if hasattr(self, '_rtstation_from'):
            return self._rtstation_from

        self._rtstation_from = None

        try:
            self._rtstation_from = RTStation.objects.filter(
                thread=self.thread,
                station=self.station_from,
                tz_departure__isnull=False
            ).order_by('-id')[0]

        except IndexError:
            if self.station_from.settlement is not None:
                try:
                    self._rtstation_from = RTStation.objects.filter(
                        thread=self.thread,
                        station__settlement=self.station_from.settlement,
                        tz_departure__isnull=False
                    ).order_by('-id')[0]

                except IndexError:
                    pass

        return self._rtstation_from

    @property
    def rtstation_to(self):
        if hasattr(self, '_rtstation_to'):
            return self._rtstation_to

        self._rtstation_to = None

        rtstation_from = self.rtstation_from

        if rtstation_from is None:
            return self._rtstation_to

        rtstation_from_id = rtstation_from.id

        try:
            self._rtstation_to = RTStation.objects.filter(
                thread=self.thread,
                station=self.station_to,
                tz_arrival__isnull=False
            ).order_by('id')[0]

        except IndexError:
            if self.station_to.settlement is not None:
                try:
                    self._rtstation_to = RTStation.objects.filter(
                        thread=self.thread,
                        station__settlement=self.station_to.settlement,
                        tz_arrival__isnull=False,
                        id__gt=rtstation_from_id
                    ).order_by('id')[0]

                except IndexError:
                    pass

        return self._rtstation_to

    def is_valid(self):
        if not RThreadSegment.is_valid(self):
            return False

        if not self.thread.runs_at(self.start_date):  # должен ходить в указанную дату
            return False

        return True

    def link(self):
        params = {
            'departure': self.start_date,
            'station_from': self.station_from.id,
            'station_to': self.station_to.id,
        }

        url = composeurl('thread', kwargs={'uid': self.thread.uid}, params=params)

        return url

    def reset(self):
        self.departure = None
        self.arrival = None
        self.direction_is_valid = False
        self.now = None
