# -*- coding: utf-8 -*-

from django.db import models
from django.utils.translation import ugettext_lazy as _

from common.models_utils.i18n import L_field, AbstractTranslate
from common.utils.fields import CodeCharField


class CoachService(models.Model):
    TANKER_L_FIELDS = ['name']

    L_name = L_field(_(u'Название'), max_length=100, add_local_field=True, default=None)

    def __unicode__(self):
        return self.L_name()

    class Meta:
        app_label = 'order'
        verbose_name = _(u'услуга')
        verbose_name_plural = _(u'услуги')


class ServiceClassNoteTranslate(AbstractTranslate):
    @classmethod
    def get_keys(cls):
        return set(ServiceClass.objects.values_list('note', flat=True))

    class Meta:
        ordering = ('value',)
        verbose_name = _(u'перевод примечания класс обслуживания')
        verbose_name_plural = _(u'переводы примечании классов обслуживания')
        app_label = 'order'


class ServiceClass(models.Model):
    owner = CodeCharField(_(u'Владелец вагона'), max_length=100, null=True)
    code = CodeCharField(_(u'Код'), max_length=100, null=False)
    is_brand = CodeCharField(_(u'Фирменный'), choices=[
        ('any', u'не важно'),
        ('yes', u'да'),
        ('no', u'нет'),
    ], max_length=100, null=False, default='any')

    name = CodeCharField(_(u'Название'), max_length=100, null=False)
    note = CodeCharField(_(u'Примечание'), max_length=100, null=True)
    L_note = ServiceClassNoteTranslate.get_L_method('note')

    services = models.ManyToManyField(CoachService, verbose_name=_(u'услуги'), blank=True)

    @classmethod
    def bind(cls, classes):
        owners = set(c.owner for c in classes)
        service_classes = set(c.service_class_code for c in classes)

        bindings = {}

        for c in cls.objects.filter(owner__in=list(owners), code__in=list(service_classes)):
            bindings.setdefault((c.owner, c.code), []).append(c)

        for c in classes:
            c.service_class = None

            for b in bindings.get((c.owner, c.service_class_code), []):
                if b.is_brand == 'any' or b.is_brand == ('yes' if c.is_brand else 'no'):
                    c.service_class = b
                    break

    def __unicode__(self):
        return "%s %s" % (self.owner, self.code)

    class Meta:
        app_label = 'order'
        verbose_name = _(u'класс обслуживания')
        verbose_name_plural = _(u'классы обслуживания')
