# -*- coding: utf-8 -*-
from copy import copy
from thread import start_new_thread

from django.conf import settings
from django.core.cache import cache

from common.models.partner import Partner
from common.utils.caching import global_cache_set

from travel.rasp.touch.order.views.train import ufs
from travel.rasp.touch.order.views.train.base import Error


def main(request, data):
    partner, cache_key, train_data = get_train_data(data, request)

    if partner not in PARTNERS:
        return {}

    if isinstance(train_data, Error):
        return {}

    context = {
        'partner': Partner.objects.get(code=partner)
    }

    context.update(class_table(partner, train_data))

    return context


def class_table(partner, data):
    if data in ('error', 'retrieving'):
        return {
            'class_table': data
            }

    new_classes = {}
    for i, klass in enumerate(data.classes):
        if klass.code not in new_classes:
            new_classes[klass.code] = klass
            order_data = klass.order_data
            order_data.update({
                'cls': i,
                'number': klass.train.number.encode('utf-8'),
                'partner': partner,
                'class': klass.code
            })

            if getattr(klass, 'ufs_car_type', None):
                order_data['car_type'] = klass.ufs_car_type.lower().encode('utf-8')

            klass.order_data = order_data
            continue

        equal_klass = new_classes[klass.code]

        merge_two_klasses(equal_klass, klass)

    new_classes = sorted(new_classes.values(), key=lambda x: x.tariff)

    return {
        'class_table': new_classes
    }


def merge_two_klasses(klass, equal_klass):
    for coach in equal_klass.coaches:
        klass.add(coach)

    if equal_klass.tariff != klass.tariff:
        klass.tariff = min(equal_klass.tariff, klass.tariff)


def get_cache_key(partner, segment):
    return settings.CACHEROOT + 'order/%s/%s/%s/%s/%s' % (
        partner,
        segment.number.replace(' ', '_'),
        segment.station_from.express_id,
        segment.station_to.express_id,
        segment.msk_departure.date(),
        )


PARTNERS = {
    'ufs': ufs
}


def get_train_data(data, request):
    segment = data['segment']

    partner = Partner.get_train_partner_code(
        segment.station_from, segment.station_to,
        request.NATIONAL_VERSION
    )

    if partner is None or partner not in PARTNERS:
        return None, None, None

    cache_key = get_cache_key(partner, segment)

    return partner, cache_key, get_data(cache_key, PARTNERS[partner].retrieve, data, request)


def get_seats_data(partner, train_cache_key, coach, from_cache=False):
    cache_key = "%s/%s/%s" % (train_cache_key, coach.klass.code, coach.number)

    if from_cache:
        return cache.get(cache_key)

    return get_data(cache_key, PARTNERS[partner].retrieve_seats, coach)


def get_data(cache_key, retrieve, *args, **kwargs):
    def retrieve_and_store():
        global_cache_set(cache_key, 'retrieving', settings.TARIFF_SUPPLIERWAIT_TIMEOUT)

        try:
            data = retrieve(*args, **kwargs)

        except Exception:
            if settings.ORDER_DEBUG:
                raise

            global_cache_set(cache_key, 'error', settings.SP_ERROR_TIMEOUT)

            return 'error'

        global_cache_set(cache_key, data, 300)

        return data

    if settings.ORDER_DEBUG:
        return retrieve_and_store()

    cached = cache.get(cache_key)

    if cached:
        return cached

    start_new_thread(retrieve_and_store, ())

    return 'retrieving'
