# -*- coding: utf-8 -*-

import base64
import logging
import re
import urllib
from urllib2 import Request

from lxml import etree

from django.conf import settings

from common.models.tariffs import Setting

from travel.rasp.touch.order.models.coachservice import ServiceClass
from travel.rasp.touch.order.views.train.base import Train, BaseKlass, Coach, Error
from travel.rasp.touch.tariffs.retrieving.base import http_retrieve
from travel.rasp.touch.tariffs.retrieving.ufs import CLASSES


SEAT_TYPES = {
    None: {
        'M5': 2,  # нижние купейные
        'M6': 3,  # верхние купейные
        'M7': 4,  # нижние боковые
        'M8': 5,  # верхние боковые
    },
    'soft': {
        'M5': 2,  # нижние купейные
        'M6': 2,  # нижние купейные
        'M7': 2,  # нижние купейные
        'M8': 2,  # нижние купейные
    }
}


log = logging.getLogger(__name__)


def retrieve(data, request):
    segment = data['segment']

    scheme = 'https' if settings.UFS_SSL else 'http'

    params = {
        'from': segment.station_from.express_id,
        'to': segment.station_to.express_id,
        'day': segment.msk_departure.day,
        'month': segment.msk_departure.month,
        'train': segment.number.encode('cp1251'),
        'terminal': settings.UFS_TERMINAL,
        'advertDomain': 'yandex.ufs-online.ru'
    }

    url = scheme + '://' + settings.UFS_HOST + settings.UFS_CARSLISTEX + '?' + urllib.urlencode(params)

    log.info('Request url: %s' % url)

    http_request = Request(url, headers={
        'Authorization': 'Basic ' + base64.b64encode(settings.UFS_USER + ':' + settings.UFS_PASSWORD),
        'User-Agent': settings.UFS_TERMINAL,
    })

    try:
        response = http_retrieve(http_request)
    except:
        log.exception(u'Ошибка запроса')

        raise

    try:
        tree = etree.parse(response)
    except:
        log.exception(u'Ошибка разбора XML')

        raise

    try:
        data = _parse_response(tree, request)
    except:
        log.exception(u'Ошибка разбора ответа UFS:\n%s' % etree.tostring(tree, encoding=unicode, pretty_print=True))

        raise

    return data



SEAT_RE = re.compile(ur'(\D*)0*(\d+)([МЖСЦИ])?$', re.U)


def _parse_response(tree, request):
    log.debug(u'Ответ UFS:\n%s' % etree.tostring(tree, encoding=unicode, pretty_print=True))

    error = tree.xpath('.//Error')

    if error:
        descr = tree.xpath('.//DescrId')[0].text

        if descr == '5064':
            return Error('too_late_or_early')
        elif descr == '5060':
            return Error('no_seats')

        log.error(
            u'Ошибка UFS (%s): %s' % (descr, tree.xpath('.//Descr')[0].text),
            extra={
                'request': request,
            }
        )

        return 'error'

    n = tree.xpath('.//N')[0]

    number = n.xpath('N1')[0].text

    second_number_node = n.xpath('N2')

    if second_number_node:
        number = second_number_node[0].text

    train = Train(number)

    kn = n.xpath('KN')[0].text.strip().split()

    is_brand = u'ФИРМ' in kn

    for ck in n.xpath('CK'):
        ufs_class_code = ck.xpath('KV')[0].text

        try:
            class_code = CLASSES[ufs_class_code]
        except KeyError:
            continue

        klass = BaseKlass(class_code)

        klass.ufs_car_type = ufs_class_code

        klass.owner = ck.xpath('VB') and ck.xpath('VB')[0].text or None

        klass.service_class_code = ck.xpath('CO') and ck.xpath('CO')[0].text or None

        klass.is_brand = is_brand

        klass.two_storey = len(ck.xpath('CV/TwoStorey'))

        klass.order_rules_data = {
            u'класс': ufs_class_code,
            u'перевозчик': klass.owner,
            u'класс обслуживания': klass.service_class_code,
        }

        if ck.xpath('MKL'):
            klass.order_data['MKL'] = ck.xpath('MKL')[0].text

        tariff = float(ck.xpath('TF')[0].text)
        other_tariff = ck.xpath('TF2') and float(ck.xpath('TF2')[0].text)

        if other_tariff and other_tariff != tariff:
            klass.tariff = min(tariff, other_tariff)
            klass.from_ = tariff != other_tariff

        else:
            klass.tariff = tariff

        klass.tariff += get_client_fee(ck)

        klass.train = train
        klass.owner = ck.xpath('VB') and ck.xpath('VB')[0].text or None
        r = ck.xpath('R')[0].text
        klass.r = r and r.split() or []

        klass.mf = u'МЖ' in klass.r
        klass.wn = u'БН' in klass.r

        tf3 = ck.xpath('TF3')

        if tf3:
            service = float(tf3[0].text)

            insurance = Setting.get('RZD_INSURANCE')

            klass.child_tariff = (klass.tariff - service - insurance) * \
                Setting.get('RZD_CHILD_TARIFF_COEFF') + \
                service + insurance
        else:
            klass.child_tariff = None

        for cv in ck.xpath('CV'):
            coach = _parse_cv(cv, klass, request)

            if coach.free:
                klass.add(coach)

        by_number = {}

        for coach in klass.coaches:
            if coach.number in by_number:
                other = by_number[coach.number]
                other.free += coach.free

                for type_, n in coach.free_by_type.items():
                    other.free_by_type[type_] = other.free_by_type.get(type_, 0) + n

                for type_, seats in coach.seats.items():
                    other.seats[type_].update(seats)
            else:
                by_number[coach.number] = coach

        klass.coaches = sorted(by_number.values(), key=lambda c: c.number)

        if klass.coaches:
            train.classes.append(klass)

    train.classes.sort(key=lambda c: c.tariff)

    ServiceClass.bind(train.classes)

    return train


def _parse_cv(cv, klass, request):
    coach = Coach()

    coach.klass = klass

    coach.two_storey = len(cv.xpath('TwoStorey'))

    coach.number = cv.xpath('VH')[0].text

    if klass.wn:
        m4 = cv.xpath('M4')

        coach.free = int(m4[0].text)

        if klass.code in ['sitting', 'common']:
            coach.free_by_type[1] = coach.free
        elif klass.code in ['soft', 'suite']:
            coach.free_by_type[2] = coach.free
    else:
        h = cv.xpath('H')

        if len(h):
            seats = h[0].text.split(', ')

            coach.free = len(seats)

            for seat in seats:
                m = SEAT_RE.match(seat)

                if m is None:
                    log.error(u'Неизвестный тип места %r' % seat, extra={
                        'request': request,
                    })

                    continue

                prefix, number, type = m.groups()

                number = prefix + number

                if type == u'М':
                    coach.seats['male'].add(number)
                elif type == u'Ж':
                    coach.seats['female'].add(number)
                elif type == u'С':
                    coach.seats['mixed'].add(number)
                elif type == u'Ц':
                    coach.seats['sex'].add(number)
                elif type == u'И':
                    # Инвалидам не можем продавать места
                    pass
                else:
                    coach.seats['mixed'].add(number)

                coach.seats['free'].add(number)

            if klass.code in ['sitting', 'common']:
                coach.free_by_type[1] = coach.free
            else:
                seat_types = SEAT_TYPES.get(klass.code, SEAT_TYPES[None])

                for code, type_ in seat_types.items():
                    element = cv.xpath(code)

                    if element:
                        n = int(element[0].text)

                        coach.free_by_type[type_] = coach.free_by_type.get(type_, 0) + n

    return coach


def get_client_fee(ck):
    client_fee_node = ck.xpath('ClientFee')
    if not client_fee_node:
        return 0.0
    return float(client_fee_node[0].attrib.get('min', 0))
