# -*- coding: utf-8

import logging
from datetime import timedelta

from django.db.models import Q

from common.models.currency import Price
from common.models.schedule import Supplier
from common.models.tariffs import ThreadTariff
from common.models.transport import TransportType
from travel.rasp.library.python.common23.date import environment
from common.utils.date import RunMask
from travel.rasp.touch.tariffs.partners.main import PARTNER_MODULES


log = logging.getLogger('touch.tariffs.bus')


def can_buy_from_static(national_version, supplier_code):
    can_buy_field = 'can_buy_%s' % national_version

    try:
        supplier = Supplier.objects.get(code=supplier_code)

    except (Supplier.DoesNotExist, Supplier.MultipleObjectsReturned):
        return False

    return getattr(supplier, can_buy_field, False)


def can_buy_from(request, supplier_code, segment,
                 order_data=None, is_static_price=True):

    segment_departure = segment.departure

    now = environment.now_aware()

    if is_static_price:
        if not can_buy_from_static(request.NATIONAL_VERSION, supplier_code):
            return False

    # RASP-9937
    if segment_departure < now:
        return False

    if supplier_code == 'avperm' and order_data:
        try:
            max_days_from_now = int(order_data['day_sale'])

            return now + timedelta(hours=2) < segment_departure < now + timedelta(days=max_days_from_now)

        except (KeyError, ValueError):
            return False

    if supplier_code == 'kvc_tula':
        return order_data and now + timedelta(hours=4, minutes=5) < segment_departure < now + timedelta(days=30)

    if supplier_code == 'Ecolines-xml':
        return now <= segment_departure

    if supplier_code == 'mrtrans':
        # RASP-15750
        start_from = (now + timedelta(hours=3))
        end_with = (now + timedelta(46)).replace(hour=0, minute=0, second=0)

        return start_from <= segment_departure < end_with

    if supplier_code == 'buscomua':
        # RASP-11127
        return segment_departure >= now + timedelta(hours=1)

    if supplier_code == 'swdfactory':
        return now + timedelta(minutes=5) < segment_departure

    if supplier_code == 'e-traffic' and order_data and 'etraffic_races' in order_data:
        dispatch_date = segment_departure.strftime('%Y-%m-%d %H:%M:%S')

        race = order_data['etraffic_races'].get(dispatch_date)

        return race

    if supplier_code == 'unistation-common-xml':
        return order_data and segment_departure > now + timedelta(hours=1) and \
            get_number_of_days_to_departure(now, segment_departure) <= 10

    if supplier_code == 'udmbus':
        return order_data and 'station_from_udm_code' in order_data and 'station_to_udm_code' in order_data

    # Для динамических цен сюда приходит partner
    if supplier_code == 'ukrmintrans_bus':
        return now + timedelta(hours=1) < segment_departure < now + timedelta(days=45)

    if supplier_code == 'Pavlodar-AV':
        return order_data and now + timedelta(minutes=15) < segment_departure < now + timedelta(days=30)

    if supplier_code == 'kamilkoc':
        try:
            key = segment_departure.strftime('%Y-%m-%d %H:%M:%S')

            sale_close_minutes = int(order_data['voyages'][key]['sale_close_minutes'])

            return order_data and \
                   now + timedelta(minutes=sale_close_minutes) < segment_departure

        except (KeyError, ValueError):
            return False

    if supplier_code == 'Donavto' and order_data:
        is_good_date = (1 <= get_number_of_days_to_departure(now, segment_departure) <= 14)

        return is_good_date and now + timedelta(hours=6) < segment_departure

    if supplier_code == 'ipektur' and order_data:
        return True

    if supplier_code == 'pamukkale' and order_data:
        return True

    if supplier_code == 'infobus' and order_data:
        if 1 <= get_number_of_days_to_departure(now, segment_departure) <= 14:
            return True

    if supplier_code == 'unitiki' and order_data:
        if get_number_of_days_to_departure(now, segment_departure) <= 90:
            if now + timedelta(minutes=5) < segment_departure:
                return True

    if supplier_code == 'takebus' and order_data:
        if get_number_of_days_to_departure(now, segment_departure) <= 30:
            if now + timedelta(minutes=60) < segment_departure:
                return True

    for module in PARTNER_MODULES.itervalues():
        can_buy_context = {
            'now': now,
            'segment': segment,
            'order_data': order_data,
            'is_static_price': is_static_price
        }
        if module.can_buy_from(request, supplier_code, can_buy_context):
            return True

    return False


def add_tariffs(segments, request=None, currency_rates=None):
    """Стоимость рейсов"""

    segments = [s for s in segments
                if s.t_type.code in ['bus', 'train'] + TransportType.WATER_TTYPE_CODES and getattr(s, 'thread', None)]

    if segments:
        tariffs_dict = get_tarriffs_dict(segments)

        for segment in segments:
            tariff_key = (segment.thread.uid, segment.station_from.id, segment.station_to.id)

            supplier_code = segment.thread.supplier.code

            if supplier_code == 'ipektur':
                segment.display_info.phone = segment.thread.company and segment.thread.company.phone or ''

            segment_tariffs = tariffs_dict.get(tariff_key, {})

            segment_tariff = None

            for year_days, mask_tariff in segment_tariffs.items():
                if RunMask.runs_at(year_days, segment.start_date):
                    segment_tariff = mask_tariff

                    break

            if segment_tariff:
                tariff = Price(segment_tariff.tariff, segment_tariff.currency)
                tariff = calc_final_price(supplier_code, tariff, currency_rates)

                order_data = segment_tariff.get_order_data()

                linked = request and can_buy_from(
                    request,
                    supplier_code,
                    segment,
                    order_data=order_data,
                    is_static_price=True
                )

                segment.display_info.set_tariff(
                    tariff,
                    linked=linked,
                    order_data=order_data,
                    is_min_tariff=segment_tariff.is_min_tariff
                )


def get_tarriffs_dict(segments):
    query = Q()
    for segment in segments:
        query |= Q(thread_uid=segment.thread.uid,
                   station_from=segment.station_from,
                   station_to=segment.station_to
                   )

    tariffs_dict = {}
    for thread_tariff in ThreadTariff.objects.filter(query):
        tariff_key = (thread_tariff.thread_uid, thread_tariff.station_from_id, thread_tariff.station_to_id)

        tariffs_dict.setdefault(tariff_key, {}).update({thread_tariff.year_days: thread_tariff})

    return tariffs_dict


def get_number_of_days_to_departure(now, departure):
    # Вычисляем количество дней до отправления в местном времени
    local_now_date = now.astimezone(departure.tzinfo).date()

    return (departure.date() - local_now_date).days


def calc_final_price(supplier_code, tariff, currency_rates):
    module = PARTNER_MODULES.get(supplier_code)

    if not hasattr(module, 'calc_final_price'):
        return tariff

    return module.calc_final_price(tariff, currency_rates)
