# -*- coding: utf-8 -*-

import datetime
import logging
import random
import time as os_time

from copy import deepcopy
from django.utils.http import urlencode

from django.conf import settings
from django.utils.translation import get_language

from common.data_api.ticket_daemon import query as td_query
from common.models.geo import Station, Settlement, Country
from common.models.partner import Partner
from common.models.transport import TransportType
from common.utils import tracer
from travel.rasp.library.python.common23.date import environment
from common.utils.caching import cache_until_switch_thread_safe, cached
from common.utils.date import KIEV_TZ, MSK_TZ
from common.utils.threadutils import run_in_thread
from common.utils.ya import has_valid_yandexuid
from common.views.tariffs import DisplayInfo

from travel.rasp.touch.tariffs.retrieving.base import Result, AllSuppliersTariffInfo, segment_data, \
    SupplierReplyTime, ExtraTrainSegment, ReplyInfo, DirectionAjaxInfo, Query, KeyReplyInfo
from travel.rasp.touch.tariffs.retrieving.ufs import UFSSeatFetcher


log = logging.getLogger('touch.seat_price')

daemon_api_query_all_log = logging.getLogger('touch.daemon_api_query_all')

agent_datetime_fmt = '%Y-%m-%dT%H:%M:%S'

def train_fetchers():
    return get_language() != 'tr' and [UFSSeatFetcher] or []


@tracer.wrap
def threaded_fetch(fetchers):
    threads = []
    results = []
    for fetcher in fetchers:
        threads.append(run_in_thread(fetcher.fetch))
    for thread in threads:
        thread.join()
        results.append(thread.get_result())
    return results


class DaemonResult(Result):
    data_types = ['seats', 'tariffs']

    def dump_prices(self):
        pass

    def __init__(self, supplier, query, data, segments, reason):
        self.supplier = supplier
        self.segments = segments

        Result.__init__(self, query, data, reason)

    @tracer.wrap
    def update_segments(self, segments, reply_time):
        if self.reason == 'success':
            self.update_extra_segments(segments)

        for field in self.data_types:
            setattr(reply_time, field, self.lmt)
            setattr(reply_time, field + '_reason', self.reason)

    def update_extra_segments(self, segments):
        self.extra_segments = self.segments.copy()

        def get_basic_key(key):
            return key

        basic_keys = dict(
            (get_basic_key(key), key)
            for key, value in self.extra_segments.items()
        )

        for s in segments:
            route_key = s.info.route_key

            self_segment = self.segments.get(route_key)

            if self_segment:
                ti_info = self.data.get(s.info.segment_key)

                if ti_info:
                    if s.info.has_info and 'bus' in ti_info.tariffs:
                        ti_info.tariffs['bus']['from'] = True

                    s.info.add_ti_info(self.supplier, ti_info)

            full_key = basic_keys.get(get_basic_key(route_key))

            if full_key and full_key in self.extra_segments:
                del self.extra_segments[full_key]

    @classmethod
    @tracer.wrap
    def make_results(cls, request, query, url, user_settlement, initiate_query=False):

        tickets_query = make_tickets_query(query, user_settlement, request.NATIONAL_VERSION)

        if initiate_query and (
            not settings.CHECK_IS_USER_VALID or has_valid_yandexuid(request)
        ):
            # FIXME: cbb
            #if settings.CHECK_IS_USER_VALID and \
            #        not cbb.validate_ipaddr(request.META.get('REMOTE_ADDR')):
            #    raise ResponseRedirectException('http://block.yandex.ru')

            try:
                tickets_query.query_all(ignore_cache=settings.IGNORE_DAEMON_CACHE)

            except Exception as e:
                daemon_api_query_all_log.exception(u'%s', repr(e))
                return []

        variants, statuses = tickets_query.collect_variants()

        results = []

        for partner_code, status in statuses.items():
            status_result = cls.make_status_result(partner_code, query, status)

            if status_result is None:
                continue

            results.append(status_result)

        for partner_code, p_variants in variants.items():
            result = cls.make_result(partner_code, query, p_variants)
            results.append(result)

        return results

    @classmethod
    def make_status_result(cls, partner_code, query, status):
        if status == 'querying' or status is None:
            return cls(partner_code, query, {}, {}, 'timeout')

        if status == 'fail':
            return cls(partner_code, query, {}, {}, 'error')

    @classmethod
    @tracer.wrap
    def make_result(cls, partner_code, query, variants):
        ti_class = cls.get_info_class()

        segments = {}
        data = {}

        for variant in variants:
            forward = variant.forward.segments[0]

            if not forward.departure or not forward.arrival:
                continue

            seats, tariffs = cls.make_seats_and_tariffs(variant, forward)

            ti = ti_class(forward.number)

            ti.seats = seats
            ti.tariffs = tariffs
            ti.et_possible = getattr(forward, 'electronic_ticket', False)

            forward.info = AllSuppliersTariffInfo(query.date, forward)

            data[forward.info.segment_key] = ti

            forward.info.add_ti_info(partner_code, ti)

            segments[forward.info.route_key] = forward

        return cls(partner_code, query, data, segments, 'success')

    @classmethod
    @tracer.wrap
    def add_tariff_and_seat(cls, request, query, url=None, user_settlement=None):
        message = u'%s - %s - %s' % (unicode(query.point_from),
                                     unicode(query.point_to), unicode(query.date))
        log.info(u'Тарифы c местами %s', message)

        in_t = cls.make_results(request, query, url, user_settlement, initiate_query=True)

        return in_t

    @classmethod
    @tracer.wrap
    def make_seats_and_tariffs(cls, variant, forward):
        deep_link = getattr(variant, 'deep_link', None)
        partner = getattr(variant, 'partner', None)
        from_company = getattr(variant, 'from_company', False)
        query_time = getattr(variant, 'query_time', 0)
        order_link = getattr(variant, 'order_link', None)
        raw_seats = getattr(variant, 'raw_seats', None)
        raw_tariffs = getattr(variant, 'raw_tariffs', None)
        raw_is_several_prices = getattr(variant, 'raw_is_several_prices', None)

        if raw_seats is not None and raw_tariffs is not None:
            seats = raw_seats
            tariffs = dict()

            for klass, tariff in raw_tariffs.iteritems():
                tariff_info = deepcopy(variant.order_data)
                tariff_info.update({
                    'price': tariff,
                    'deep_link': deep_link,
                    'partner': partner,
                    'order_link': order_link,
                    'from_company': from_company,
                    'query_time': query_time
                })

                if raw_is_several_prices and raw_is_several_prices.get(klass, False):
                    tariff_info.update({'from': True})

                if forward.supplier_code:
                    tariff_info['supplier'] = forward.supplier_code

                tariffs[klass] = tariff_info

        else:
            seats = {cls.ticket_cls: getattr(variant, 'seats', 1)}
            tariffs = {cls.ticket_cls: variant.order_data}

            tariff_info = variant.order_data
            tariff_info.update({
                'price': variant.tariff,
                'deep_link': deep_link,
                'partner': partner,
                'order_link': order_link,
                'from_company': from_company,
                'query_time': query_time
            })

            if forward.supplier_code:
                tariff_info['supplier'] = forward.supplier_code

        return seats, tariffs


class PlaneResult(DaemonResult):
    key_of_delay = 'plane_'
    ticket_cls = 'economy'

    # скорей всего для других типов транспорта тоже должен быть общий AllSuppliersTariffInfo
    # и null должен расцениваться как done. Но я боюсь трогать что-то кроме авиа.
    @classmethod
    @tracer.wrap
    def make_results(cls, request, query, url, user_settlement, initiate_query=False):

        tickets_query = make_tickets_query(query, user_settlement, request.NATIONAL_VERSION)

        if initiate_query and (
            not settings.CHECK_IS_USER_VALID or has_valid_yandexuid(request)
        ):
            try:
                # опрашиваем только партнёров расписаний
                tickets_query.query_all(ignore_cache=settings.IGNORE_DAEMON_CACHE)

            except Exception as e:
                daemon_api_query_all_log.exception(u'%s', repr(e))

                # нужно проставить статусы
                return []

        variants, statuses = tickets_query.collect_variants()

        results = []

        for partner_code, status in statuses.items():
            # неответ для расписаний - это корректный статус
            if not status:
                status = 'done'

            status_result = cls.make_status_result(partner_code, query, status)

            if status_result is None:
                continue

            results.append(status_result)

        all_suppliers_tariff_info = {}

        for partner_code, p_variants in variants.items():
            result = cls.make_result(partner_code, query, p_variants, all_suppliers_tariff_info)
            results.append(result)

        return results

    @classmethod
    @tracer.wrap
    def make_result(cls, partner_code, query, variants, all_suppliers_tariff_info):
        ti_class = cls.get_info_class()

        segments = {}
        data = {}

        for variant in variants:
            forward = variant.forward.segments[0]

            if not forward.departure or not forward.arrival:
                continue

            seats, tariffs = cls.make_seats_and_tariffs(variant, forward)

            ti = ti_class(forward.number)

            ti.seats = seats
            ti.tariffs = tariffs
            ti.et_possible = getattr(forward, 'electronic_ticket', False)

            key = (query.date, forward)

            if key not in all_suppliers_tariff_info:
                all_suppliers_tariff_info[key] = AllSuppliersTariffInfo(query.date, forward)

            forward.info = all_suppliers_tariff_info[key]

            data[forward.info.segment_key] = ti

            forward.info.add_ti_info(partner_code, ti)

            segments[forward.info.route_key] = forward

        return cls(partner_code, query, data, segments, 'success')

    def update_extra_segments(self, segments):
        self.extra_segments = self.segments.copy()

        for s in segments:
            route_key = s.info.route_key

            self_segment = self.segments.get(route_key)

            # если нашли соответствующий сегмент из базы
            # и времена прибытия и отправления совпадают
            if self_segment and segment_data(self_segment) == segment_data(s):
                ti_info = self.data.get(s.info.segment_key)

                if ti_info:
                    s.info.add_ti_info(self.supplier, ti_info)

                if route_key in self.extra_segments:
                    del self.extra_segments[route_key]


class BusResult(DaemonResult):
    key_of_delay = 'bus_'
    ticket_cls = 'bus'


class TrainResult(DaemonResult):
    key_of_delay = 'train_'
    ticket_cls = 'train'


class BlablacarResult(DaemonResult):
    key_of_delay = 'blablacar_'
    ticket_cls = 'blablacar'

    def on_success(self):
        for tariff_info in self.data.values():
            order_data = tariff_info.tariffs[self.ticket_cls]
            url_params = order_data['url_params']

            # https://st.yandex-team.ru/TOUCHRASP-1525
            url_params.update({
                'utm_campaign': 'RU_YANDEXRASP_PSGR_TRASP_none',
                'comuto_cmkt': 'RU_YANDEXRASP_PSGR_TRASP_none'
            })

            url_params[order_data['from_param_name']] = self.get_full_settlement_title(self.query.point_from)
            url_params[order_data['to_param_name']] = self.get_full_settlement_title(self.query.point_to)
            str_params = urlencode(url_params)
            order_data['url'] = order_data['base_url'] + str_params
            order_data['m_url'] = order_data['base_m_url'] + str_params

    def get_full_settlement_title(self, settlement):
        omonim_title = settlement.L_omonim_title(get_language(), False)
        return (omonim_title['add'] + u', ' if omonim_title['add'] else u'') + omonim_title['title']


@tracer.wrap
def make_tickets_query(query, user_settlement, national_version):
    q = td_query.Query(
        user_settlement=user_settlement,
        point_from=query.point_from,
        point_to=query.point_to,
        date_forward=query.date,
        date_backward=None,
        passengers={'adults': 1},
        klass='economy',
        national_version=national_version,
        t_code=query.t_type.code
    )
    return q


@cache_until_switch_thread_safe
def find_express_station(point):
    if isinstance(point, Station):
        return point if point.express_id else None

    if not isinstance(point, Settlement):
        return None

    train_type = TransportType.objects.get(code='train')

    pseudo_stations = point.station_set.filter(t_type=train_type,
                                               express_id__isnull=False,
                                               majority__code='express_fake')

    if pseudo_stations:
        return pseudo_stations[0]

    stations = point.station_set.filter(
        t_type=train_type, hidden=False, express_id__isnull=False
    ).order_by('majority__id')

    if stations:
        return stations[0]

    return None


@tracer.wrap
def get_filtered_train_fetchers(express_query, fetcher_classes):
    fetchers = []
    for f in fetcher_classes:
        if not f.can_be_asked():
            log.debug(u"Не спрашиваем %s", f.__name__)
            continue

        if not f.is_suitable(express_query):
            log.debug(u"Не подходин при данных условиях запроса %s",f.__name__)
            continue

        log.debug(u"Спрашиваем %s", f.__name__)
        fetchers.append(f)

    return fetchers


@tracer.wrap
def express_direction_key(express_query, fetcher_classes):
    fetchers = get_filtered_train_fetchers(express_query, fetcher_classes)
    key = u'_'.join([express_query.point_from.express_id,
                     express_query.point_to.express_id,
                     str(express_query.date)] + map(lambda f: f.__class__.__name__,
                     fetchers))
    return key


@cached(express_direction_key, settings.CACHES['default']['LONG_TIMEOUT'])
@tracer.wrap
def get_express_direction_fetchers(express_query, fetcher_classes):
    fetchers = get_filtered_train_fetchers(express_query, fetcher_classes)
    if len(fetchers) > 1:
        random.seed()
        fetchers.pop(random.randint(0,1))
    return fetchers


def get_possible_train_fetcher_classess(query):
    fetchers = []

    express_query = get_express_query(query)

    if express_query:
        for fetcher in train_fetchers():
            fetchers.append((express_query, fetcher))

    return fetchers


@tracer.wrap
def get_suitable_train_fetcher_classess(query, fetcher_classes):
    express_query = get_express_query(query)
    if express_query is None:
        raise StopIteration

    fetchers = get_filtered_train_fetchers(express_query, fetcher_classes)
    error = False
    if not express_query.is_order and not settings.ALWAYS_ASK_ALL:
        cached_fetchers = []
        for f in fetchers:
            result = f.get_cache(express_query)
            response_reason = result and result.reason
            if response_reason == 'error':
                error = True
            elif response_reason == 'success':
                cached_fetchers.append(f)

        if cached_fetchers:
            fetchers = cached_fetchers
        elif not error:
            fetchers = get_express_direction_fetchers(express_query, fetcher_classes)

    if fetchers:
        log.debug(u"Спрашиваем %s", u", ".join(f.supplier for f in fetchers))
    else:
        log.debug(u"Никого из поездов не спрашиваем")

    for fetcher_class in fetchers:
        yield express_query, fetcher_class


@tracer.wrap
def get_express_query(query):
    station_from = find_express_station(query.point_from)
    station_to = find_express_station(query.point_to)
    if station_from and station_to:
        log.info(u'В точке %s, взяли станцию %s',
                 unicode(query.point_from), unicode(station_from))
        log.info(u'В точке %s, взяли станцию %s',
                 unicode(query.point_to), unicode(station_to))

        express_query = query.copy()
        express_query.initial_point_from = query.point_from
        express_query.initial_point_to = query.point_to
        express_query.point_from = station_from
        express_query.point_to = station_to
        return express_query
    else:
        direction = u'%s - %s' % (unicode(query.point_from),
                                     unicode(query.point_to))
        log.warning(u'Не нашли станций с express_id по направлению %s',
                    direction)

@tracer.wrap
def add_tariff_train(query, partner):
    """ Данные по наличию мест поездов
    """

    log.info(u'Грузим данные по поездам')

    query.route_numbers = dict([(s.number[:-1], s.number)
                                for s in query.segments])

    fetcher_classes = []

    if partner == 'ufs':
        fetcher_classes = [UFSSeatFetcher]

    elif partner == 'tickets_ua':
        fetcher_classes = []

    fetchers = []
    for related_query, fetcher_class in get_suitable_train_fetcher_classess(query, fetcher_classes):

        fetchers.append(fetcher_class(related_query))

    results = threaded_fetch(fetchers)

    return results


@tracer.wrap
def tariffs_date(point_from, departure, t_type):
    if t_type in ['plane']:
        return departure.date()

    if t_type == 'train':
        if point_from.country_id == Country.UKRAINE_ID:
            return departure.astimezone(KIEV_TZ).date()
        else:
            return departure.astimezone(MSK_TZ).date()

    return departure.date()


def segment_tariffs_date(segment):
    return tariffs_date(segment.station_from, segment.departure, segment.t_type.code)


# point это либо станция, либо город, поскольку набор полей частично совпадает,
# в этом скрипте можно использовать любой из объектов
@tracer.wrap
def add_availability_info(request, segments, point_from, point_to, is_order=False, uri=None,
                          supplement=tuple(), early_border=None, late_border=None,
                          user_settlement=None, allow_blablacar=False):
    """
    segments - список объектов типа RThreadSegment
    point_from, point_to объекты отправления и прибытия
    """

    if not settings.SEATS_FETCH_ENABLED:
        return None, None, dict()

    start = os_time.time()
    log.info(u'Обновляем данные')
    train_groups = {}
    plane_groups = {}
    bus_groups = {}
    blablacar_groups = {}

    fill_groups_from_segments(segments, train_groups, plane_groups, bus_groups)

    fill_groups_from_supplement(supplement, point_from, point_to, early_border,
                                late_border, train_groups, plane_groups, bus_groups, blablacar_groups)

    # Запускаем каждую группу в отдельном потоке
    threads = []

    run_plane_groups_in_threads(request, threads, plane_groups, point_from, point_to,
                                is_order, uri, user_settlement)

    run_train_groups_in_threads(request, threads, train_groups, point_from, point_to,
                                is_order, uri, user_settlement)

    # Ждем завершения работы потоков
    reply_info_by_key = {}

    extra_segments = {}

    for thr in threads:
        tracer.join_thread(thr)

        try:
            for result in thr.get_result():
                key = result.get_key()

                supplier = result.supplier

                key_supplier_time = reply_info_by_key.setdefault(key, {})

                reply_time = key_supplier_time.setdefault(supplier, SupplierReplyTime())

                result.update_segments(segments, reply_time)

                if hasattr(result, 'extra_segments'):
                    extra_segments.update(result.extra_segments)

        except Exception:
            log.exception(u'Один из потоков завершился некорректно')

            if settings.DEBUG:
                raise

    local_now = point_from.localize(msk=datetime.datetime.now())

    early_border = early_border and max(local_now - datetime.timedelta(hours=1), early_border)

    if supplement:
        extra_segments = ExtraTrainSegment.limit(extra_segments, (early_border, late_border))
        ExtraTrainSegment.fill_titles(extra_segments)
        ExtraTrainSegment.correct_stations(extra_segments, point_from, point_to, uri=uri)
    else:
        extra_segments = {}

    log.info(u'Суммарные данные обновлены за %f секунд, ' + \
             u' всего групп под датам %d, запущено потоков %d',
             os_time.time() - start, len(train_groups) + len(plane_groups),
             len(threads))

    reply_info = ReplyInfo(reply_info_by_key)

    ajax_info = DirectionAjaxInfo(point_from, point_to, reply_info, segments,
                                  early_border, late_border)

    fix_extra_segments(extra_segments)

    return ajax_info, reply_info, extra_segments


def fix_extra_segments(extra_segments):
    """ Хак, для отображения рейсов пополнения """

    for s in extra_segments.values():
        if not hasattr(s, 'display_info'):
            s.display_info = DisplayInfo()


def fill_groups_from_segments(segments, train_groups, plane_groups, bus_groups):
    for segment in segments:
        msk_departure_date = segment.departure.astimezone(MSK_TZ).date()

        departure_date = segment_tariffs_date(segment)

        if segment.t_type.code == 'plane':
            plane_groups.setdefault((msk_departure_date, departure_date), []).append(segment)

        if segment.t_type.code == 'bus':
            bus_groups.setdefault((msk_departure_date, departure_date), []).append(segment)

        if segment.t_type.code == 'train':
            train_groups.setdefault((msk_departure_date, departure_date), []).append(segment)

        segment.info = AllSuppliersTariffInfo(departure_date, segment)


def fill_groups_from_supplement(supplement, point_from, point_to,
                                early_border, late_border,
                                train_groups, plane_groups, bus_groups, blablacar_groups):
    for t_type in supplement:
        msk_departure_date = early_border.date()
        departure_date = tariffs_date(point_from, early_border, t_type)
        end_date = tariffs_date(point_to, late_border, t_type)

        blablacar_groups.setdefault(departure_date, [])

        while departure_date <= end_date:
            if t_type == 'train':
                train_groups.setdefault((msk_departure_date, departure_date), [])

            if t_type == 'plane':
                plane_groups.setdefault((msk_departure_date, departure_date), [])

            if t_type == 'bus':
                bus_groups.setdefault((msk_departure_date, departure_date), [])

            departure_date += datetime.timedelta(days=1)


def run_plane_groups_in_threads(request, threads, plane_groups, point_from, point_to,
                                is_order, uri, user_settlement):
    local_today = point_from.localize(msk=environment.now()).date()

    for (msk_dep_date, departure_date), plane_segments in plane_groups.items():
        # Местная текущая дата-время в городе из которого выезжаем
        # Местная дата-время планируемого выезда
        days_from_today = (departure_date - local_today).days

        # Если это уже в далеком прошлом не получаем данные
        if is_the_plane_distant_past(days_from_today):
            continue

        cache_timeout = get_plane_cache_timeout(days_from_today)
        query = Query(segments=plane_segments, point_from=point_from, point_to=point_to,
                      date=departure_date, cache_timeout=cache_timeout,
                      t_type=TransportType.objects.get(code='plane'))

        log.info(u'Обновляем самолеты')
        thr = run_in_thread(PlaneResult.add_tariff_and_seat, args=(request, query, uri, user_settlement))

        threads.append(thr)


def run_train_groups_in_threads(request, threads, groups, point_from, point_to,
                                is_order, uri, user_settlement):
    if get_language() == 'tr':
        return

    partner_code = Partner.get_train_partner_code(point_from, point_to, request.NATIONAL_VERSION)

    local_today = point_from.localize(msk=environment.now()).date()

    ## Информация по поездам
    for (msk_dep_date, departure_date), train_segments in groups.items():
        days_from_today = (departure_date - local_today).days

        # Если это уже в далеком прошлом не получаем данные
        if is_the_train_distant_past(days_from_today):
            continue

        query = Query(segments=train_segments, date=departure_date,
                      point_from=point_from, point_to=point_to,
                      is_order=is_order, t_type=TransportType.objects.get(code='train'))

        # FixMe: Find more suitable place for setting cache_timeout
        if partner_code == 'ukrmintrans_train':
            query.cache_timeout = settings.UKRMINTRANS_TRAIN_CACHE_TIMEOUT

        # запрос без демона
        thr = run_in_thread(add_tariff_train, args=(query, partner_code))
        threads.append(thr)


def get_plane_cache_timeout(days_from_today):
    # 6 часов для рейсов на даты, отстоящие от текущей более чем на 2 месяца,
    if days_from_today > 60:
        cache_timeout = 6 * 60 * 60
    # 3 часа для рейсов на даты, отстоящие от текущей более чем на 1 месяц,
    elif days_from_today > 30 and days_from_today <= 60:
        cache_timeout = 3 * 60 * 60
    # 1 час для рейсов на даты, отстоящие от текущей более чем на 1 неделю,
    elif days_from_today > 7 and days_from_today <= 30:
        cache_timeout = 60 * 60
    # 30 минут для рейсов на даты, отстоящие от текущей более чем на 4 дня,
    elif days_from_today > 4 and days_from_today <= 7:
        cache_timeout = 30 * 60
    # 15 минут для всех остальных.
    else:
        cache_timeout = 15 * 60

    return cache_timeout


def is_the_plane_distant_past(days_from_today):
    return days_from_today < -settings.SEATS_IN_PAST_DAY


def is_the_bus_distant_past(days_from_today):
    return days_from_today < -settings.SEATS_IN_PAST_DAY


def is_the_train_distant_past(days_from_today):
    return days_from_today < -settings.SEATS_IN_PAST_DAY


class FakeSegment(object):
    pass


def get_search_info(request, point_from, point_to, t_code, date_,
                    is_order=False, ready_keys=None, segments=None, url=None, user_settlement=None):
    routes = {}
    reply_info = {}
    results = []

    query = Query(
        point_from=point_from,
        point_to=point_to,
        date=date_,
        is_order=is_order,
        t_type=TransportType.objects.get(code=t_code),
    )

    if segments:
        query.segments = segments

        for segment in segments:
            segment.info = AllSuppliersTariffInfo(query.date, segment)

    if t_code == 'plane':
        results.extend(PlaneResult.make_results(request, query, url, user_settlement))
    elif t_code == 'train':
        # Результаты запроса без демона
        for related_query, fetcher_class in get_possible_train_fetcher_classess(query):
            results.append(fetcher_class.get_cache(related_query))

    results = filter(lambda r: r is not None, results)

    segment_keys = set()

    if not segments:
        segments = []

        for r in results:
            if r.reason == "success":
                for segment_key in r.data:
                    segment_keys.add(segment_key)

        for segment_key in segment_keys:
            info = AllSuppliersTariffInfo(query.date, segment_key=segment_key)

            if ready_keys is None or info.route_key in ready_keys:
                s = FakeSegment()
                s.info = info
                s.data = ready_keys.get(info.route_key) if ready_keys is not None else None
                segments.append(s)

    extra_segments = {}

    for r in results:
        supplier = r.supplier
        reply_time = reply_info.setdefault(supplier, SupplierReplyTime())
        r.update_segments(segments, reply_time)

        if hasattr(r, 'extra_segments'):
            extra_segments.update(r.extra_segments)

    for s in segments:
        routes[s.info.segment_key] = s.info

    return routes, extra_segments, KeyReplyInfo(reply_info)
