# -*- coding: utf-8 -*-

import json
import logging
import time as time_module
from datetime import datetime, time, timedelta

from django.conf import settings
from django.core.cache import cache
from django.db.models import Q
from django.template.loader import render_to_string
from django.utils.http import urlencode
from django.utils.translation import get_language

from common.dynamic_settings.default import conf
from common.models.currency import Price
from common.models.geo import Settlement, Station
from common.models.schedule import RTStation
from common.models.special_tariffs import CacheSet
from common.models.tariffs import CLS_NAMES, CLS_MAPPING, NotTariffTrain, AeroexTariff
from common.utils import tracer
from travel.rasp.library.python.common23.date import environment
from common.utils import request_helper
from common.utils.date import MSK_TZ
from common.utils.locations import set_lang_param
from common.views.tariffs import DisplayInfo
from common.xgettext.i18n import gettext

from travel.rasp.touch.tariffs.retrieving import add_availability_info, get_search_info
from travel.rasp.touch.tariffs.utils import can_buy_from
from travel.rasp.touch.tariffs import bus as bus_tariffs


log = logging.getLogger('touch.tariffs')


class CookedInfo(object):
    dynamic = True

    def __init__(self, t_type, departure_date):
        self.places = []
        self.t_type = t_type
        self.departure_date = departure_date
        self.can_buy = False
        self.et_marker = False
        self.min_tariff = None
        self.has_data = True
        self.yamoney = False
        self.timestamp = None
        self.no_data_no_seats = False
        self.wait = False

    def link(self, request, segment, buy_link_func,
             point_from=None, point_to=None, extra_params={}):

        if get_language() != 'tr' and segment.t_type.code == 'plane' and \
                point_from and point_to:

            params = {
                'fromId': point_from.point_key,
                'toId': point_to.point_key,
                'when': segment.departure.date(),
                'forward': segment.number,
            }

            return '/ticket/order/?' + urlencode(dict((k, v) for k, v in params.iteritems() if v))

        params = self.get_params(segment, point_from, point_to)

        params.update(extra_params)

        return buy_link_func(request, params)

    def get_params(self, segment, point_from, point_to):
        params = {
            'station_from': segment.station_from.id,
            'station_to': segment.station_to.id,
            'departure': segment.msk_departure.replace(tzinfo=None),
            'arrival': segment.msk_arrival.replace(tzinfo=None),
            'title': segment.title,
            'date': segment.departure.date(),
            'number': segment.number or segment.thread.hidden_number or '',
            't_type': segment.t_type.code,
        }

        if self.et_marker:
            params['et'] = 't'

        if getattr(segment, 'thread', None):
            params['thread'] = segment.thread.uid

        if point_from and point_to:
            params.update({
                'point_from': point_from.point_key,
                'point_to': point_to.point_key,
            })

        return params

    def copy(self):
        new = CookedInfo(self.t_type, self.departure_date)
        new.can_buy = self.can_buy
        new.et_marker = self.et_marker
        new.min_tariff = self.min_tariff
        new.has_data = self.has_data
        new.yamoney = self.yamoney
        new.timestamp = self.timestamp
        new.no_data_no_seats = self.no_data_no_seats

        return new

    def json(self):
        return {
            'places': [p.json() for p in self.places],
            'et_marker': self.et_marker,
        }

    @property
    def request_key(self):
        return '%s_%s' % (self.t_type, self.departure_date)

    @property
    def ndns_class(self):
        if self.no_data_no_seats:
            return ' js-ndns-%s' % self.request_key

        return ''


class TariffPlace(object):
    """Тариф для отображения"""

    def __init__(self, cooked, code, name, type_, timestamp,
                 tariff=None, min_=False, seats=0,
                 partner_code=None, deep_link=None,
                 order_link=None, partner=None,
                 from_company=False, query_time=0):

        self.cooked = cooked
        self.code = code
        self.name = name
        self.type_ = type_
        self.tariff = tariff if hasattr(tariff, 'currency') else Price(tariff)
        self.min = min_
        self.timestamp = timestamp
        self.partner_code = partner_code
        self.seats = seats
        self.deep_link = deep_link
        self.partner = partner
        self.from_company = from_company
        self.query_time = query_time
        self.order_link = order_link

    def link(self, request, segment, buy_link_func, point_from=None, point_to=None):

        params = {
            'cls': self.code,
            'tariff': self.tariff.value,
        }

        if segment:
            return self.cooked.link(request, segment, buy_link_func, point_from, point_to, params)

        return '?' + set_lang_param(params)

    @property
    def yamoney(self):
        return self.type_ in ['discount', 'roundtrip_discount']

    def json(self):
        tariff = {
            'sort_value': self.tariff.sort_value,
            'value': self.tariff.value,
            'currency': self.tariff.currency,
            }

        return [self.name, self.code, tariff, self.min]

    def __repr__(self):
        return "<TariffPlace code=%r tariff=%r>" % (self.code, self.tariff)


class TariffChoice(TariffPlace):
    def __init__(self, order_data, cooked, code, name, type_, timestamp, tariff=None, min_=False, seats=0):
        self.order_data = order_data

        TariffPlace.__init__(self, cooked, code, name, type_, timestamp, tariff, min_, seats)

    def link(self, request, segment, buy_link_func, point_from=None, point_to=None):
        order_data = self.order_data.copy()

        order_data.update({
            'tariff': self.tariff.value,
            'currency': self.tariff.currency,
        })

        # Импортируем здесь, так как весь этот класс - хак
        from common.utils.order_data import signed_order_data

        params = {'choice': urlencode(signed_order_data(order_data))}

        if segment:
            return self.cooked.link(request, segment, buy_link_func, point_from, point_to, params)

        return '?' + set_lang_param(params)


class ClsInfo(object):
    def __init__(self, seats, tariff, timestamp, order_data=None, partner_code=None):
        self.seats = seats
        self.tariff = tariff
        self.timestamp = timestamp
        self.order_data = order_data
        self.partner_code = partner_code

    def __repr__(self):
        return "<ClsInfo seats=%r tariff=%r timestamp=%r>" % (self.seats, self.tariff, self.timestamp)

    def places(self, cooked, code, name):
        if self.seats and self.tariff:
            if self.order_data:
                yield TariffChoice(
                    self.order_data,
                    cooked, code, name, 'price',
                    self.timestamp,
                    self.tariff['price'],
                    self.tariff.get('from'),
                    seats=self.seats
                )

            else:
                yield TariffPlace(
                    cooked, code, name, 'price',
                    self.timestamp,
                    self.tariff['price'],
                    self.tariff.get('from'),
                    seats=self.seats,
                    partner_code=self.partner_code,
                    partner=self.tariff.get('partner'),
                    deep_link=self.tariff.get('deep_link'),
                    order_link=self.tariff.get('order_link'),
                    from_company=self.tariff.get('from_company', False),
                    query_time=self.tariff.get('query_time', 0)
                )


def check_if_can_buy(raw, min_tariff, client_city, today_plane, time_left, point_from, point_to):
    if not min_tariff:
        return False

    for supplier, info in raw.by_supplier.items():
        if not (info.tariffs_time or info.roundtrip_tariffs_time):
            continue

        if can_buy_from(supplier, info, client_city, today_plane, time_left, point_from, point_to):
            return True


def tariff_worse_then(current, other):
    if current is None:
        return True

    if current['price'] < other['price']:
        return False

    if current['price'] > other['price']:
        return True

    # тарифы от авиакомпаний лучше, чем от агенств
    if current.get('from_company') != other.get('from_company'):
        if current.get('from_company') is True:
            return False
        if other.get('from_company') is True:
            return True

    # кто раньше ответил, тот и молодец
    if current.get('query_time') <= current.get('query_time'):
        return False
    return True


def process_info(raw, roundtrip=False):
    by_cls = {}

    for supplier, info in raw.by_supplier.items():
        seats = info.seats
        tariffs = info.tariffs
        seats_time = info.seats_time
        tariffs_time = info.tariffs_time

        if not seats:
            continue

        for code, available in seats.items():
            tariff = tariffs and tariffs.get(code)

            # При наличии мест обязательно нужны тарифы
            if available:
                if tariff:
                    # И учитываем время приход тарифов
                    timestamp = tariffs_time

                else:
                    # При отсутствии тарифов игнорируем информацию
                    continue

            else:
                # При отсутствии мест важно только время прихода этой информации
                timestamp = seats_time

            try:
                current_tariff = by_cls[code].tariff
            except KeyError:
                current_tariff = None

            # Записываем данные первого полностью ответившего поставщика
            if tariff_worse_then(current_tariff, tariff):
                order_data = None

                if supplier == 'biletall':
                    url = tariff.get('url')

                    if url:
                        order_data = {
                            'url': url,
                            'partner': supplier,
                            'ucus': tariff['ucus'],
                        }

                by_cls[code] = ClsInfo(available, tariff, timestamp, order_data, supplier)

    return by_cls


def find_min_tariff(by_cls):
    # Минимальная цена, в том числе для неотображаемых классов
    min_tariff = None
    min_tariff_ts = None
    min_tariff_seats = None

    # Проходим по всем классам
    for code, info in by_cls.items():
        # если код есть в нашем маппинге и есть места
        if code in CLS_NAMES and info.seats:
            try:
                # то цену учитываем
                tariff = info.tariff

                if tariff and tariff['price']:
                    if min_tariff is None or tariff['price'] < min_tariff:
                        min_tariff = tariff['price']
                        min_tariff_ts = info.timestamp
                        min_tariff_seats = info.seats
                        break

            except KeyError:
                pass

    return min_tariff, min_tariff_ts, min_tariff_seats


def cook_tariffs(raw, t_type, reply_info=None, key_reply_info=None, client_city=None,
                 no_data_no_seats=False, today_plane=False, time_left=None, point_from=None, point_to=None):
    """
    Подготовка данных с информацией о местах и ценах для отображения на странице
    """

    log.debug('raw: %r' % raw)

    # Для агентов Серёжа не хочет отдавать места "обратно"
    if 'agent' in raw.by_supplier:
        agent_info = raw.by_supplier['agent']
        agent_info.roundtrip_seats = agent_info.seats
        agent_info.roundtrip_seats_time = agent_info.seats_time

    cooked = CookedInfo(t_type, raw.departure_date)

    if key_reply_info is None:
        try:
            key_reply_info = reply_info.by_key[cooked.request_key]
        except KeyError:
            return cooked

    cooked.wait = not key_reply_info.complete

    cooked.no_data_no_seats = no_data_no_seats

    if t_type == 'blablacar':
        cooked.url = raw.by_supplier['blablacar'].tariffs['blablacar']['url']
        cooked.m_url = raw.by_supplier['blablacar'].tariffs['blablacar']['m_url']

    # Убираем поставщиков, у которых нельзя купить
    raw.by_supplier = dict((supplier, info) for supplier, info in raw.by_supplier.items()
                           if can_buy_from(supplier, info, client_city, today_plane, time_left, point_from, point_to))

    log.debug('raw ready to process: %r' % raw)

    by_cls = process_info(raw)

    log.debug("by_cls: %r" % by_cls)

    # Есть электронный билет
    et_possible = any(i.et_possible for i in raw.by_supplier.values())

    min_tariff, min_tariff_ts, _ = find_min_tariff(by_cls)

    if t_type in ['train', 'bus'] and et_possible:
        cooked.et_marker = True

    # Преобразовываем словарь в пригодный для отображения
    for name, code in CLS_MAPPING:
        if code in by_cls:
            cooked.places.extend(by_cls[code].places(cooked, code, name))

    log.debug("places: %r" % cooked.places)

    cooked.can_buy = check_if_can_buy(raw, min_tariff, client_city, today_plane, time_left, point_from, point_to)
    cooked.min_tariff = min_tariff
    cooked.yamoney = any(p.yamoney for p in cooked.places)
    cooked.timestamp = max(p.timestamp for p in cooked.places) if cooked.places else None

    return cooked


class MobileClsInfo(object):
    timestamp = None

    def __init__(self, seats, tariff):
        self.seats = seats
        self.tariff = tariff


def process_info_mobile(raw):
    """Минимальные цены по классам"""

    by_cls = {}

    for info in raw.by_supplier.values():
        # Нет информации по местам, продолжаем
        if not info.seats:
            continue

        for code, available in info.seats.items():
            if not available:
                by_cls[code] = MobileClsInfo(available, None)
                continue

            if not info.tariffs:
                continue

            try:
                tariff = info.tariffs[code]
            except KeyError:
                # Места без тарифов не нужны
                continue

            try:
                current_tariff = by_cls[code].tariff
            except KeyError:
                current_tariff = None

            if not current_tariff or tariff['price'] < current_tariff['price']:
                by_cls[code] = MobileClsInfo(available, tariff)

    return by_cls


def cook_mobile_tariffs(segment, t_type, reply_info, no_data_no_seats=False):
    """Выбирает минимальную цену. Не отображает "нет мест" """
    raw = segment.info
    log.debug('raw: %r' % raw)

    cooked = CookedInfo(t_type, raw.departure_date)

    # Уже присутствующие тарифы
    if 'tariffs_info' in segment.display_info:
        cooked.places.append((None, 0, segment.display_info['tariffs_info'].places[0].tariff))

    try:
        key_reply_info = reply_info.by_key[cooked.request_key]
    except KeyError:
        cooked.some_data = True
        cooked.complete = True
        return cooked

    cooked.complete = key_reply_info.complete
    cooked.no_seats = raw.by_supplier and all(s.seats and not any(s.seats.values()) for s in raw.by_supplier.values())


    by_cls = process_info_mobile(raw)

    log.debug("by_cls: %r" % by_cls)

    min_tariff, _, min_tariff_seats = find_min_tariff(by_cls)

    # Преобразовываем словарь в пригодный для отображения
    for name, code in CLS_MAPPING:
        if code in by_cls:
            if by_cls[code].seats:
                cooked.places.append((name, by_cls[code].seats, by_cls[code].tariff and by_cls[code].tariff['price']))

    # Если нечего показывать
    if not cooked.places:
        # Если есть минимальный тариф
        if min_tariff:
            # Показываем его
            cooked.places = [(None, min_tariff_seats, min_tariff)]
        else:
            # Если ответ от UFS или МобиТикета пришёл и данных по местам нет,
            # показываем "нет мест" для определенных поездов
            if no_data_no_seats and key_reply_info.seats_success:
                cooked.no_seats = True

    cooked.some_data = cooked.places or cooked.no_seats

    return cooked


def planes_departure_time(point, uids, local_date):
    result = dict()

    if isinstance(point, Settlement):
        rtstations = RTStation.objects.filter(
            station__settlement=point,
            thread__route__route_uid__in=uids,
            tz_departure__isnull=False
        )

    elif isinstance(point, Station):
        rtstations = RTStation.objects.filter(
            station=point,
            thread__route__route_uid__in=uids,
            tz_departure__isnull=False
        )

    else:
        raise TypeError("Unsupported point type %r" % type(point))

    # Начало суток местного времени aware
    local_day_start_dt = point.localize(datetime.combine(local_date, time(0, 0)))

    for rts in rtstations:
        naive_start_dt = datetime.combine(local_date, rts.thread.tz_start_time)

        # Время отправления с начальной станции aware
        departure_dt = rts.get_departure_dt(naive_start_dt, out_tz=rts.station.pytz)

        # если время вылета не в эту дату, то добавляем сутки
        if departure_dt < local_day_start_dt:
            departure_dt = rts.get_departure_dt(naive_start_dt + timedelta(days=1), out_tz=rts.station.pytz)

        # Время отправления с данной станции по Москве
        result[rts.thread.route.route_uid] = departure_dt.astimezone(MSK_TZ)

    return result


def search_tariffs(request, currency_info, point_from, point_to, keys, ready_keys, request_timestamp, client_city, url=None):
    # Время текущего обновления
    response_timestamp = time_module.time()

    # Обновления мест и цен
    updates = {}
    complete_keys = []
    ndns_keys = []

    local_now = point_from.localize(msk=environment.now())

    local_date_from = local_now.date()

    extra_segments = {}

    # Список дат с типами транспорта
    for key in keys:
        t_type, dt = key.split('_')

        dt = datetime.strptime(dt, '%Y-%m-%d').date()

        routes, key_extra_segments, key_reply_info = get_search_info(request, point_from, point_to, t_type, dt, None, ready_keys=ready_keys, url=url, user_settlement=client_city)

        rebase_raw_tariffs(routes.values(), currency_info)
        rebase_raw_tariffs((s.info for s in key_extra_segments.values()), currency_info)

        extra_segments.update(key_extra_segments)

        if t_type == 'plane' and dt - local_date_from < timedelta(days=2): # Не нужно все это спрашивать для рейсов, которые точно летят далеко в будущем
            cache_key = "/".join([settings.CACHEROOT, 'departure_times', point_from.point_key, point_to.point_key, key])
            cached = cache.get(cache_key)

            if cached:
                departure_times = cached
            else:
                departure_times = planes_departure_time(point_from, routes.keys(), dt)
                cache.set(cache_key, departure_times, settings.CACHES['default']['LONG_TIMEOUT'])
        else:
            departure_times = dict()

        today_plane = t_type == 'plane' and dt == local_date_from

        log.debug("reply_info: %r" % key_reply_info)

        if key_reply_info.complete:
            complete_keys.append(key)

        if key_reply_info.seats_success:
            ndns_keys.append(key)

        for uid, raw in routes.items():
            departure = departure_times.get(uid)

            if departure is None and ready_keys:
                ready_key = ready_keys.get(raw.route_key)

                if ready_key and len(ready_key) == 8:
                    dep = datetime.strptime(dt.strftime('%Y%m%d') + ready_key[:4], '%Y%m%d%H%M')
                    arr = datetime.strptime(dt.strftime('%Y%m%d') + ready_key[4:], '%Y%m%d%H%M')

                    if arr < dep:
                        arr += timedelta(1)

                    departure = point_from.localize(loc=dep)

            time_left = departure - local_now if departure else None

            cooked = cook_tariffs(raw, t_type,
                                  key_reply_info=key_reply_info,
                                  client_city=client_city,
                                  today_plane=today_plane,
                                  time_left=time_left,
                                  point_from=point_from,
                                  point_to=point_to)

            # Возвращаем апдейт, если информация по рейсу обновилась
            if cooked.timestamp > request_timestamp:
                updates[raw.route_key] = cooked

        for segment in key_extra_segments.values():
            time_left = segment.departure - environment.now_aware()

            segment.display_info = DisplayInfo()

            segment.display_info['tariffs_info'] = cook_tariffs(segment.info, t_type,
                                                                key_reply_info=key_reply_info,
                                                                client_city=client_city,
                                                                today_plane=today_plane,
                                                                time_left=time_left,
                                                                point_from=point_from,
                                                                point_to=point_to)

    return {
        'ts': response_timestamp,
        'updates': updates,
        'complete': complete_keys,
        'ndns': ndns_keys,
    }, updates, extra_segments


def free_seats_classes():
    return [
        # Cидячие места
        ('01', ['common', 'sitting'], gettext(u'сидячие')),
        ('02', ['platzkarte'], gettext(u'плацкарт')),
        ('03', ['compartment'], gettext(u'купе')),
        # Спальный вагон
        ('04', ['suite', 'soft'], gettext(u'СВ')),
    ]


def free_seats_margins():
    return [
        (50, 'green', gettext(u'есть')),
        (10, 'yellow', gettext(u'мало')),
        (1, 'orange', gettext(u'заканчиваются')),
        (0, 'red', gettext(u'нет')),
    ]


def holiday_tickets(cacheset_title=''):
    u"""Наличие мест на новогодние праздники"""

    try:
        cacheset = CacheSet.objects.get(title=cacheset_title)
    except CacheSet.DoesNotExist:
        return ''

    block = []

    seats_margins = free_seats_margins()
    seats_classes = free_seats_classes()

    for direction in cacheset.directions.all():
        if not direction.data:
            continue

        data = json.loads(direction.data)

        row = []

        for d in cacheset.dates:
            free_by_class = data.get(str(d), {})

            free_bar = []

            for code, classes, name in seats_classes:
                free = 0

                for cls in classes:
                    free += free_by_class.get(cls, 0)

                for margin, color, desc in seats_margins:
                    if free >= margin:
                        break

                img = "%si/holiday_tickets/%s_%s.png" % (
                    settings.MARKUP_URL, code, color[0]
                )

                free_bar.append((name, desc, img))

            link = '/search/train/?' + urlencode({
                'fromId': direction.point_from_key,
                'toId': direction.point_to_key,
                'when': str(d),
            })

            row.append((link, free_bar))

        block.append((direction.title, direction.alt_link, row,))

    context = {
        'table': block,
        'dates':cacheset.dates
    }

    return render_to_string('info/holiday_tickets.html', context)


@tracer.wrap
def fill_tariffs_info(segments, currency_info, supplement, point_from, point_to, when, allow_blablacar, request):
    # Подготовка информации о местах и ценах для отображения и сортировки

    ajax_tariffs_info, reply_info, extra_segments = add_availability_info(
        request,
        segments,
        point_from,
        point_to,
        uri=request_helper.build_absolute_uri(request),
        supplement=supplement,
        early_border=point_from.localize(loc=datetime.combine(when, time(0, 0))),
        late_border=point_from.localize(loc=datetime.combine(when + timedelta(days=1), time(4, 0))),
        user_settlement=request.client_city,
        allow_blablacar=allow_blablacar
    )

    segments[:] = [s for s in segments if s.info and s.info.route_key not in extra_segments]

    segments.extend(extra_segments.values())

    local_date_from = point_from.localize(request.now).date()

    rebase_raw_tariffs((s.info for s in segments), currency_info)

    for segment in segments:
        if segment.gone:
            continue

        # Для поездов добавляем флаг, если при отсутствии данных нужно
        # показывать "нет мест"

        no_data_no_seats = False

        if is_our_direction(point_from, point_to):
            if (segment.t_type.code == 'train'
                and when < (request.now.date() + timedelta(days=conf.TRAIN_ORDER_DEFAULT_DEPTH_OF_SALES))  # RASP-2078
                and segment.number not in NotTariffTrain.number_set() # RASP-3193
            ):
                no_data_no_seats = True

        today_plane = segment.t_type.code in ['plane', 'bus'] and \
            segment.departure.date() == local_date_from

        departure_dt = segment.departure

        # Для сегментов из демона приходит naive datetime
        if departure_dt.tzinfo is None:
            departure_dt = MSK_TZ.localize(departure_dt)

        segment.display_info['tariffs_info'] = cook_tariffs(
            segment.info,
            segment.t_type.code,
            reply_info=reply_info,
            client_city=request.client_city,
            no_data_no_seats=no_data_no_seats,
            today_plane=today_plane,
            time_left=departure_dt - environment.now_aware(),
            point_from=segment.station_from
        )

    # Добавляем статические тарифы, получается они на данные момент конфликтуют с динамикой
    # FIXME: разрешить конфликты статических и динамических тарифов
    bus_tariffs.add_tariffs(segments, request, currency_info.rates)

    # Добавляем тарифы электричечного типа
    add_station_station_tariffs(segments, add_tariffs_to_gone=True, currency_info=currency_info)

    return ajax_tariffs_info


def get_segment_tariff_type_code(segment):
    thread = getattr(segment, 'thread', None)

    if thread is None:
        return None

    tariff_type = thread.tariff_type

    if tariff_type is None:
        if segment.t_type.code == 'suburban':
            if segment.thread.is_express or segment.thread.is_aeroexpress:
                return 'express'

            return 'etrain'

        return None

    return tariff_type.code


def add_station_station_tariffs(segments, add_tariffs_to_gone=False, currency_info=None):
    """Тарифы точка-точка из таблицы aeroextariffs"""
    # RASP-3104

    # Сегменты по парам станций отправления-прибытия
    pairs = {}

    for segment in segments:
        if segment.gone and not add_tariffs_to_gone:
            continue

        tariff_type_code = get_segment_tariff_type_code(segment)

        if tariff_type_code is None:
            continue

        pairs.setdefault((segment.station_from, segment.station_to), []).append((segment, tariff_type_code))

    if pairs:
        for (station_from, station_to), pair_segments in pairs.items():
            tariffs = get_station_station_tariffs(station_from, station_to)

            for segment, tariff_type_code in pair_segments:
                if tariff_type_code in tariffs:
                    segment.display_info.set_tariff(tariffs[tariff_type_code].tariff)
                    segment.display_info.set_aux_tariffs(tariffs, tariff_type_code)

    if currency_info:
        rebase_tariffs(segments, currency_info)


def is_our_direction(point_from, point_to):
    country_from_id = point_from.country_id
    country_to_id = point_to.country_id

    our_direction = False

    if country_from_id in settings.OUR_COUNTRIES and country_to_id in settings.OUR_COUNTRIES:
        our_direction = True

    # Для маршрутов, станция отправления которых лежит в Украине,
    # и для которых УФС не вернул данных, в результатах большого
    # поиска нужно писать "нет данных" вместо нет мест. (c) zharik
    if country_from_id == 187: # RASP-3236
        our_direction = False

    return our_direction


def get_suburban_tariffs(segments):
    """Тарифы для электричечного поиска"""

    if not segments:
        return None, None

    # Сортируем по важности станций
    segments.sort(key=lambda s: (s.station_from.majority_id, s.station_to.majority_id))

    # Пытаемся найти первую нитку без изменений
    try:
        segment = (s for s in segments if not s.thread.type.code == 'change').next()
    except StopIteration:
        # не нашли, берем первый
        segment = segments[0]

    return get_suburban_tariffs_for_segment(segment)


def get_station_station_tariff_qs(station_from, station_to):
    """Условия для выборки тарифа"""

    return AeroexTariff.objects.filter(
        Q(station_from=station_from, station_to=station_to) |
        Q(station_to=station_from, station_from=station_to, reverse=True)
    ).order_by('-precalc')  # Чтобы ручные имели приоритета над предрасчитанными


def get_suburban_tariffs_for_segment(segment):
    tariffs = get_station_station_tariffs(segment.station_from, segment.station_to)

    if 'express' in tariffs and tariffs['express'].suburban_search:
        main_tariff = tariffs['express']
    else:
        main_tariff = tariffs.get('etrain')

    if not main_tariff:
        return None, None

    tariffs_by_cat = {}

    if len(tariffs) == 1:
        return main_tariff.tariff, None

    for t in sorted(tariffs.values(), key=lambda t: t.type.order):
        tariffs_by_cat.setdefault(t.type.category, []).append(t)

    return main_tariff.tariff, tariffs_by_cat


def get_station_station_tariffs(station_from, station_to):
    tariffs_qs = get_station_station_tariff_qs(station_from, station_to)

    tariffs = {}

    for t in tariffs_qs:
        t.tariff = Price(t.tariff, t.currency or 'RUR')

        tariffs[t.type.code] = t

    return tariffs


def rebase_raw_tariffs(infos, currency_info):
    for i in infos:
        for info in i.by_supplier.values():
            for tariff in info.tariffs.values():
                try:
                    price = tariff['price']

                    if isinstance(price, Price):
                        price.rebase(currency_info.rates)
                except KeyError:
                    pass


def rebase_tariffs(segment, currency_info):
    for s in segment:
        tariffs_info = s.display_info.tariffs_info

        if tariffs_info:
            for place in tariffs_info.places:
                place.tariff.rebase(currency_info.rates)

