# -*- coding: utf-8 -*-

from datetime import datetime, timedelta, time


import re

from common.models.schedule import TrainSchedulePlan
from travel.rasp.library.python.common23.date import environment
from common.utils.text import NBSP
from common.xgettext.i18n import xgettext
from route_search.by_number.helpers import number_variants
from stationschedule.type.suburban import SuburbanSchedule
from stationschedule.type.tablo import TabloRealSchedule
from stationschedule.views import get_schedule_class


def get_schedule(context, schedule_type, t_type_code, station, start_date, end_date, event, mod, limit=None):
    schedule_class = get_schedule_class(station, schedule_type=schedule_type, t_type_code=t_type_code)

    schedule_class_params = {
        'limit': limit,
        'station': station,
        'event': event,
        't_type_code': t_type_code
    }

    if schedule_type == 'tablo':
        schedule_class_params['terminal'] = context['terminal']
        if schedule_class == TabloRealSchedule and context['query']:  # фильтр по запросу в статусном табло работает неправильно
            schedule_class_params['query'] = context['query']
    elif schedule_type == 'suburban':
        direction_code = SuburbanSchedule.DIRECTION_ALL if not station.get_direction() else context['direction_code']

        schedule_class_params.update({
            'requested_direction': direction_code,
            'city': context['city'],
            'event': None
        })

    schedule = schedule_class(**schedule_class_params)

    if mod != 'all':
        schedule = schedule.build(start_datetime_limit=start_date,
                                  end_datetime_limit=end_date)
    else:
        schedule = schedule.build(start_time_limit=start_date.time(), end_time_limit=end_date.time())

    if schedule_type == 'schedule':
        def filter_raw_schedule_route(items):
            for item in items:
                if item.cancel:
                    continue
                if event == 'departure' and item.is_last_station:
                    continue

                yield item

        schedule.add_raw_schedule_route_filter(filter_raw_schedule_route)

    if context['query'] and schedule_class != TabloRealSchedule:
        def query_filter(schedule_route_gen):
            for schedule_route in schedule_route_gen:
                if _like_query(context['query'], schedule_route):
                    yield schedule_route

        schedule.add_schedule_route_filter(query_filter)

    if mod == 'all':
        def sort_raw_schedule_route(items):
            return sorted(items, key=lambda schedule_route: schedule_route.event_dt.time())

        schedule.add_raw_schedule_route_filter(sort_raw_schedule_route)

    return schedule


def get_start_and_end(station, mod):
    local_now = environment.now_aware().astimezone(station.pytz)

    if mod == 'nearest':
        return local_now, datetime.combine(local_now, time.max)

    if mod == 'tomorrow':
        start_date = local_now + timedelta(days=1)
        return datetime.combine(start_date, time.min), datetime.combine(start_date, time.max)

    # all, today строятся на день
    start_date = datetime.combine(local_now, time.min)
    return start_date, datetime.combine(start_date, time.max)


def get_init_schedule(context, station, event, mod, schedule_type=None, t_type_code=None, limit=None):

    start, end = get_start_and_end(station=station,
                                   mod=mod)

    return get_schedule(context=context,
                        event=event,
                        schedule_type=schedule_type,
                        t_type_code=t_type_code,
                        station=station,
                        start_date=start,
                        end_date=end,
                        mod=mod, limit=limit), start, end


def time_text(item, filter_flag):
        text = []
        stops = None

        stops_text = item.schedule and item.schedule.L_stops()

        if stops_text:
            stops = stops_text

            # после цифр вместо пробела пишем &nbsp;
            stops = re.sub(r'\d+ ',
                           lambda obj: obj.group(0).replace(' ', NBSP),
                           stops,
                           flags=re.U)

            text += [xgettext(u'остановки: <stops/>', stops=stops), ' ']

        if filter_flag == 'all':
            now = environment.now_aware()
            _, next_plan = TrainSchedulePlan.get_current_and_next(today=now.date())
            days = item.L_days_text(next_plan=next_plan)

            if days:
                text += ['<br>', days] if stops else [days]

        return text


def train_time_text(item, filter_flag):
        text = []
        stops = None

        stops_text = item.schedule and item.schedule.L_stops()

        if stops_text:
            stops = stops_text

            # после цифр вместо пробела пишем &nbsp;
            stops = re.sub(r'\d+ ',
                           lambda obj: obj.group(0).replace(' ', NBSP),
                           stops,
                           flags=re.U)

            text += [xgettext(u'остановки: <stops/>', stops=stops), ' ']

        days = item.L_days_text()

        if filter_flag == 'all' and days:
            if stops:
                text += ['<br>']

            text += [days]

        return text

# копи паст из statusTablo, так как нужен поиск по всем расписаниям


def _like_query(query, schedule_route):
    for field_value in _search_field_values(schedule_route):
        try:
            if field_value.contains(query):
                return True

        except AttributeError:
            if query in field_value:
                return True

    return False


def _search_field_values(schedule_route):
    if schedule_route.number:
        yield schedule_route.number

    yield schedule_route.L_title()

    yield u"%s %s" % (schedule_route.number, schedule_route.L_title())

    next_station = schedule_route.rtstation.next_station

    if next_station:
        yield next_station.L_title

    prev_station = schedule_route.rtstation.prev_station

    if prev_station:
        yield prev_station.L_title

    if schedule_route.company:
        yield schedule_route.company.L_title

    if next_station and next_station.country:
        yield next_station.country.L_title

    for number in number_variants(schedule_route.number):
        yield number
