# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import datetime, timedelta
from logging import getLogger, Logger

import requests
from django.conf import settings
from django.utils.translation import get_language
from requests import PreparedRequest
from typing import Tuple

from common.models.currency import Price
from common.models.geo import Station
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.rasp.touch.tariffs.utils import CLS_NAMES
from travel.rasp.touch.touch.core.lib.train_tariff_key_map_fabric import TrainTariffKeyMapFabric, train_tariff_key_map_fabric


define_setting('TRAIN_API_URL', {
    Configuration.PRODUCTION: 'https://production.train-api.rasp.internal.yandex.net',
    Configuration.TESTING: 'https://testing.train-api.rasp.internal.yandex.net',
    Configuration.DEVELOPMENT: 'https://testing.train-api.rasp.internal.yandex.net'
}, default=None)

define_setting('TRAINS_FRONT_URL', {
    Configuration.PRODUCTION: 'https://trains.yandex.ru',
    Configuration.TESTING: 'https://testing.train.common.yandex.ru',
    Configuration.DEVELOPMENT: 'https://testing.train.common.yandex.ru'
}, default=None)


class TrainTariffResult(object):
    def __init__(self, records, train_order_url_owner, is_electronic_ticket, querying):
        # type: (Tuple[TrainTariffRecord, ...], unicode, bool, bool) -> None
        self.records = records
        self.train_order_url_owner = train_order_url_owner
        self.is_electronic_ticket = is_electronic_ticket
        self.querying = querying

    def __repr__(self):
        return '<TrainTariffResult records={} train_order_url_owner={} is_electronic_ticket={} querying={}/>'.format(
            self.records,
            self.train_order_url_owner,
            self.is_electronic_ticket,
            self.querying
        )

    def __eq__(self, other):
        if not isinstance(other, TrainTariffResult):
            return False

        return (
                self.records == other.records and
                self.train_order_url_owner == other.train_order_url_owner and
                self.is_electronic_ticket == other.is_electronic_ticket and
                self.querying == other.querying
        )


class TrainTariffRecord(object):
    def __init__(self, coach_type, coach_title, train_order_url_owner, train_order_url, price, seats, lower_seats,
                 upper_seats, several_prices):
        # type: (unicode, unicode, unicode, unicode, Price, int, int, int, bool) -> None
        self.coach_type = coach_type
        self.coach_title = coach_title
        self.train_order_url_owner = train_order_url_owner
        self.train_order_url = train_order_url
        self.price = price
        self.seats = seats
        self.lower_seats = lower_seats
        self.upper_seats = upper_seats
        self.several_prices = several_prices

    def __repr__(self):
        return (
            '<TrainTariffRecord coach_type={} train_owner={} '
            'order_url={} price={} seats={}/{}/{} several_prices={}/>'.format(
                self.coach_type,
                self.train_order_url_owner,
                self.train_order_url,
                self.price,
                self.seats,
                self.lower_seats,
                self.upper_seats,
                self.several_prices
            ))

    def __eq__(self, other):
        if not isinstance(other, TrainTariffRecord):
            return False

        return (
                self.coach_type == other.coach_type and
                self.coach_title == other.coach_title and
                self.train_order_url_owner == other.train_order_url_owner and
                self.train_order_url == other.train_order_url and
                self.price == other.price and
                self.seats == other.seats and
                self.lower_seats == other.lower_seats and
                self.upper_seats == other.upper_seats and
                self.several_prices == other.several_prices
        )


class TrainTariffApiClient(object):
    UNKNOWN_OWNER = 'unknown'

    def __init__(self, host, transport, train_tariff_key_map_fabric, logger):
        # type: (unicode, any, TrainTariffKeyMapFabric, Logger) -> None
        self.host = host
        self.transport = transport
        self.train_tariff_key_map_fabric = train_tariff_key_map_fabric
        self.logger = logger

    def find_tariffs_for(self, from_station, to_station, departure_dt, number, national_version, rates, init_query):
        # type: (Station, Station, datetime, unicode, unicode, dict, bool) -> TrainTariffResult
        departure_date = departure_dt.date()
        str_departure = departure_date.strftime('%Y-%m-%d')
        self.logger.info(
            'Try to find info about: from=%d to=%d departure=%s, national_version=%s',
            from_station.id, to_station.id, str_departure, national_version
        )
        try:
            poll_part = 'poll/'
            if init_query:
                poll_part = ''

            response = self.transport.get("{}/{}/api/segments/train-tariffs/{}?".format(
                self.host, get_language(), poll_part
            ), params={
                'pointFrom': from_station.point_key,
                'pointTo': to_station.point_key,
                'national_version': national_version,
                'includePriceFee': 1,
                'startTime': (departure_dt - timedelta(days=1)).isoformat(),
                'endTime': (departure_dt + timedelta(days=1)).isoformat(),
                'partner': 'im'
            }, timeout=2)
            response.raise_for_status()
        except requests.RequestException:
            self.logger.exception(
                'Can not fetch info for: '
                'from=%d to=%d departure=%s, national_version=%s',
                from_station.id, to_station.id, str_departure, national_version
            )
            return TrainTariffResult([], self.UNKNOWN_OWNER, False, False)

        data = response.json()

        self.logger.info(
            'Receive info from backend about %d segments and requiests is %s for: '
            'from=%d to=%d departure=%s, national_version=%s',
            len(data['segments']), data['querying'], from_station.id, to_station.id, str_departure,
            national_version
        )

        keys = self.train_tariff_key_map_fabric.generate_map_keys(departure_dt, number)
        self.logger.info(
            'Filter records by keys=%s stationFrom=%d stationTo=%d',
            ', '.join(sorted(keys)), from_station.id, to_station.id
        )

        records, is_electronic_ticket, train_order_url_owner = next(
            (
                self._parse_tariffs(s['tariffs'], rates) for s in data['segments']
                if (any(s['key'].startswith(key) for key in keys) and
                    s['stationFrom']['id'] == from_station.id and
                    s['stationTo']['id'] == to_station.id
                    )
            ),
            ([], False, self.UNKNOWN_OWNER)
        )

        return TrainTariffResult(
            records=records,
            train_order_url_owner=train_order_url_owner,
            is_electronic_ticket=is_electronic_ticket,
            querying=data['querying'] and not records
        )

    def _parse_price(self, raw_price, rates):
        currency_code = 'RUR' if raw_price['currency'] == 'RUB' else raw_price['currency']
        p = Price(
            int(round(raw_price['value'])),
            currency_code
        )

        p.rebase(rates)
        return p

    def _parse_tariffs(self, tariffs, rates):
        is_electronic_ticket = tariffs.get('electronicTicket', False)

        records = tuple(sorted(
            (
                TrainTariffRecord(
                    coach_type=code,
                    coach_title=CLS_NAMES.get(code, code),
                    train_order_url_owner=record['trainOrderUrlOwner'],
                    train_order_url=make_full_train_order_url(record['trainOrderUrl']),
                    price=self._parse_price(record['price'], rates),
                    seats=record['seats'],
                    lower_seats=record.get('lowerSeats', 0),
                    upper_seats=record.get('upperSeats', 0),
                    several_prices=record.get('severalPrices')
                ) for code, record in tariffs['classes'].items()
            ),
            key=lambda r: r.price
        ))

        return records, is_electronic_ticket, next((r.train_order_url_owner for r in records), self.UNKNOWN_OWNER)


def make_full_train_order_url(url):
    if url.startswith('/'):
        url = settings.TRAINS_FRONT_URL + url
    req = PreparedRequest()
    req.prepare_url(url, {'utm_source': 'rasp', 'utm_medium': 'thread_button'})
    return req.url


train_tariff_api_client = TrainTariffApiClient(
    host=settings.TRAIN_API_URL,
    transport=requests,
    train_tariff_key_map_fabric=train_tariff_key_map_fabric,
    logger=getLogger(__name__)
)
