# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import timedelta

from travel.rasp.touch.touch.core.lib.train_number import (
    TrainNumberParser, TrainNumberReverser, train_number_parser, train_number_reverser
)


class TrainTariffKeyMapFabric(object):
    """
    Фабрика по генерации ключей для маппинга тарифов из сервиса поездов с сегментами из расписаний
    по номеру рейса и дате отправления поезда
    """
    TIME_RANGE_HOURS = 2
    TRAIN_KEY_FORMAT = 'train {} {}'

    def __init__(self, train_number_parser, train_number_reverser):
        # type: (TrainNumberParser, TrainNumberReverser) -> None
        self._train_number_parser = train_number_parser
        self._train_number_reverser = train_number_reverser

    def generate_map_keys(self, departure_dt, raw_train_number):
        normalized_train_number = set(self._get_possible_numbers(raw_train_number))

        dt_keys = {
            self._make_datetime_key(departure_dt - timedelta(hours=self.TIME_RANGE_HOURS / 2)),
            self._make_precise_datetime_key(departure_dt),
            self._make_datetime_key(departure_dt + timedelta(hours=self.TIME_RANGE_HOURS / 2))
        }

        return {
            self.TRAIN_KEY_FORMAT.format(number, dt_key)
            for number in normalized_train_number
            for dt_key in sorted(dt_keys)
        }

    def _get_possible_numbers(self, raw_train_number):
        train_number = self._train_number_parser.parse(raw_train_number)
        if train_number is None:
            yield raw_train_number
            return
        yield train_number.serialize()
        yield self._train_number_reverser.reverse(train_number).serialize()

    def _make_precise_datetime_key(self, dt):
        return dt.strftime('%Y%m%d_%H%M')

    def _make_datetime_key(self, dt):
        aligned_dt = dt.replace(hour=int(dt.hour // self.TIME_RANGE_HOURS) * self.TIME_RANGE_HOURS)
        return aligned_dt.strftime('%Y%m%d_%H')


train_tariff_key_map_fabric = TrainTariffKeyMapFabric(
    train_number_parser=train_number_parser,
    train_number_reverser=train_number_reverser
)
