# -*- coding: utf8 -*-

from __future__ import unicode_literals

import cgi
import os
import urlparse
from calendar import monthrange
from datetime import timedelta
from dateutil.relativedelta import relativedelta

from django.conf import settings
from django.utils.translation import get_language

from common.utils import bemhtml
from common.utils.locations import langify
from common.utils.bemhtml.template import Template
from common.utils.date import uni_strftime
from common.utils.ya import get_change_language_url, get_country_by_lang
from common.xgettext.i18n import tgettext, gettext, xgettext

from travel.rasp.touch.touch.core.templates.blocks import metrika
from travel.rasp.touch.touch.core.templates.utils import tune_desktop_alternate_link, desktop_version_link
from travel.rasp.touch.touch.core.templates.utils.favicons import get_favicons
from travel.rasp.touch.touch.core.templates.utils.tizer import teaser_info
from travel.rasp.touch.touch.core.templates.utils.touch_links import feedback_url, user_agreement_url, auth_url
from travel.rasp.touch.touch.core.helpers.alternate_domains import get_alternate_domains


BUNDLES_PATH = os.path.join(settings.PROJECT_PATH, 'markup', 'desktop.bundles')

LOCAL_PORTAL_SERVICES_URLS = {
    'maps': {
        'ru': 'https://maps.yandex.ru/',
        'ua': 'https://maps.yandex.ua/',
        'by': 'https://maps.yandex.by/',
        'kz': 'https://maps.yandex.kz/',
        'uz': 'https://maps.yandex.ru/',
        'com.tr': 'https://harita.yandex.com.tr/'
    },
    'images': {
        'ru': 'https://yandex.ru/images/touch/',
        'ua': 'https://yandex.ua/images/touch/',
        'by': 'https://yandex.by/images/touch/',
        'kz': 'https://yandex.kz/images/touch/',
        'uz': 'https://yandex.uz/images/touch/',
        'com.tr': 'https://yandex.com.tr/gorsel/touch/'
    },
    'mail': {domain: 'https://mail.yandex.%s/lite/inbox' % domain for domain in settings.TOUCH_NATIONAL_DOMAINS},
    'market': {
        'ru': 'https://m.market.yandex.ru/',
        'ua': 'https://m.market.yandex.ua/',
        'by': 'https://m.market.yandex.by/',
        'kz': 'https://m.market.yandex.kz/',
        'uz': 'https://m.market.yandex.ru/',
        'com.tr': 'https://market.yandex.com.tr'
    }
}


class Base(Template):
    page = None
    show_calendar = False
    share = False

    def render_bemjson(self, bemjson):
        return bemhtml.render(path=os.path.join(BUNDLES_PATH, self.page, '%s.all.priv.js' % self.page),
                              bemjson=bemjson,
                              lang=get_language())

    @property
    def request(self):
        return self.context.request

    def static_url(self, path):
        return settings.MARKUP_URL + path

    def asset_url(self, extension):
        path = 'markup/desktop.bundles/%s/_%s.%s' % (self.page, self.page, extension)

        return self.static_url(path)

    def make_alternate_link(self, alternate):
        full_path = self.request.get_full_path()
        alt_link = 'https://{0}{1}'.format(settings.NATIONAL_DOMAIN_MAP[alternate['domain']], full_path)
        return langify(alt_link, lang=alternate['lang'], national_version=alternate['domain'])

    def alternative_links(self):
        alternatives = get_alternate_domains()
        return [(self.make_alternate_link(alternate), alternate['hreflang'])
                for alternate in alternatives]

    def teaser(self, code):
        return self.context.teasers and self.context.teasers.render_teaser(code, self) or None

    def alarm(self):
        teaser = self.teaser('ahtung')

        if not teaser:
            return

        return {
            'block': 'b-alarm',
            'content': [{
                'elem': 'string',
                'content': teaser
            }]
        }

    def b_footer(self):
        desktop_version_url = tune_desktop_alternate_link(self.request)
        return {
            'block': 'b-footer',
            'content': [
                {
                    'block': 'b-copyright',
                    'start': '2008',
                    'content': {
                        'block': 'b-link',
                        'content': gettext('Яндекс'),
                        'url': 'https://www.yandex.%s' % self.request.tld
                    }
                },
                {
                    'elem': 'info',
                    'content': {
                        'block': 'b-link',
                        'url': feedback_url(self.request),
                        'attrs': metrika.reach_goal_on('click', 'FEEDBACK'),
                        'content': gettext('Обратная связь')
                    }
                },
                {
                    'elem': 'info',
                    'content': {
                        'block': 'b-link',
                        'url': user_agreement_url(self.request),
                        'content': gettext('Пользовательское соглашение')
                    }
                },
            ]
        }

    def b_lang_switcher(self):
        langs = self.request.supported_langs

        if len(langs) < 2:
            return

        lang = get_language()

        country_code = get_country_by_lang(lang)

        return {
            'block': 'b-lang-switcher',
            'js': True,
            'tag': 'span',
            'content': [
                {
                    'elem': 'lang',
                    'tag': 'span',
                    'content': country_code
                },
                {
                    'block': 'b-icon',
                    'mods': {'type': 'flag'},
                    'src': '//yastatic.net/lego/2.9-7/blocks-desktop/b-country-flag/_size-16/b-country-flag_size-16_%s.png' % country_code.lower()
                },
                {
                    'elem': 'change',
                    'tag': 'span',
                    'content': gettext('сменить язык')
                }
            ]
        }

    def b_head(self):
        request = self.request
        yauser = request.yauser

        user_content = []

        if yauser.is_authenticated():
            if yauser.fields['social']:
                user_content = [{'elem': 'icon', 'src': self.static_url('i/avatar.png')}]

            else:
                user_content = [{'elem': 'name', 'content': cgi.escape(yauser.fields['display_name'])}]

        if self.request.tld == 'com.tr':
            logo_mods = {'lang': 'tr'}

        else:
            logo_mods = None

        return {
            'block': 'touch-head',
            'content': [
                {
                    'elem': 'col',
                    'elemMods': {'type': 'user'},
                    'content': [
                        {
                            'block': 'touch-user',
                            'content': user_content
                        },
                        {
                            'elem': 'btn',
                            'mix': [{'block': 'b-sideslide', 'elem': 'switcher'}]
                        }
                    ]
                },
                {
                    'elem': 'col',
                    'elemMods': {'type': 'logo'},
                    'attrs': metrika.reach_goal_on(action='click', target='GO-TO-MAIN-YANDEX-PAGE'),
                    'content': {'elem': 'logo', 'mods': logo_mods, 'url': 'https://www.yandex.%s' % self.request.tld}
                },
                Base.b_touch_head_service(self.context.get('is_main', False))
            ]
        }

    @staticmethod
    def b_touch_head_service(is_main):
        content = {
            'elem': 'service',
            'elemMods': {'lang': get_language()},
            'content': gettext('Расписания')
        }
        if not is_main:
            content['url'] = langify('/')

        return {
            'elem': 'col',
            'elemMods': {'type': 'service'},
            'attrs': metrika.reach_goal_on(action='click', target='GO-TO-MAIN-RASP-PAGE') if not is_main else None,
            'content': content
        }

    def content(self):
        raise NotImplementedError

    def page_title(self):
        return self.context.page_title

    def month_switcher(self, step):
        today = self.context.form.loc_today
        start_dt = today - timedelta(days=3)
        dt = start_dt + relativedelta(months=step)

        if start_dt.year != dt.year:
            dt_format = '%B %Y'

        else:
            dt_format = '%B'

        return {
            'block': 'b-month-switcher',
            'js': True,
            'mods': {'state': step and 'hidden' or 'current'},
            'content': [
                {
                    'elem': 'switch',
                    'elemMods': {'disabled': not step and 'yes' or None, 'arrow-type': 'prev'},
                    'content': {
                        'elem': 'icon',
                        'elemMods': {'direction': 'left'}
                    }
                },
                {
                    'elem': 'month-year',
                    'content': uni_strftime(dt_format, dt).capitalize()
                },
                {
                    'elem': 'switch',
                    'elemMods': {'disabled': step == 11 and 'yes' or None, 'arrow-type': 'next'},
                    'content': {
                        'elem': 'icon',
                        'elemMods': {'direction': 'right'}
                    }
                }
            ]
        }

    def month_calendar(self, step):
        form = self.context.form

        today = form.loc_today
        start_dt = today - timedelta(days=3)
        end_dt = today + relativedelta(months=11)
        dt = start_dt + relativedelta(months=step)
        month_start = dt - timedelta(days=dt.day - 1)
        weekday, ndays = monthrange(month_start.year, month_start.month)
        date = form['when'].data or today

        rows = []
        week = dict((i, None) for i in xrange(7))
        d = None

        for c in xrange(ndays):
            d = month_start + timedelta(days=c)
            week[d.weekday()] = d

            if d.weekday() == 6:
                rows.append(week)
                week = dict((i, None) for i in xrange(7))

        if d.weekday() != 6:
            rows.append(week)

        def _js_params(day):
            if not day:
                return {
                    'day': None,
                    'delta': 0
                }

            if today.year != day.year:
                value = uni_strftime('%d %B %Y', day)

            else:
                value = uni_strftime('%d %B', day)

            return {
                'day': value,
                'delta': (day - today).days
            }

        def _mods(i, day):
            mods = {}

            if i > 4:
                mods['week-end'] = 'yes'

            if day == date:
                mods['checked'] = 'yes'

            if not day:
                mods['active'] = 'no'

            if not day or day < start_dt or day > end_dt:
                mods['active'] = 'no'

            return mods

        weeks = [{
            'elem': 'week',
            'js': True,
            'content': [{
                'elem': 'date-day',
                'js': _js_params(day),
                'elemMods': _mods(i, day),
                'content': getattr(day, 'day', '')
            } for i, day in week.iteritems()]
        } for week in rows]

        return {
            'block': 'b-calendar',
            'mods': {'state': step and 'hidden' or 'current'},
            'js': True,
            'content': [
                {
                    'elem': 'days-week',
                    'content': [
                        {
                            'elem': 'day-week',
                            'content': xgettext('пн')
                        },
                        {
                            'elem': 'day-week',
                            'content': xgettext('вт')
                        },
                        {
                            'elem': 'day-week',
                            'content': xgettext('ср')
                        },
                        {
                            'elem': 'day-week',
                            'content': xgettext('чт')
                        },
                        {
                            'elem': 'day-week',
                            'content': xgettext('пт')
                        },
                        {
                            'elem': 'day-week',
                            'elemMods': {'week-end': 'yes'},
                            'content': xgettext('сб')
                        },
                        {
                            'elem': 'day-week',
                            'elemMods': {'week-end': 'yes'},
                            'content': xgettext('вс')
                        },
                    ]
                }
            ] + weeks
        }

    def content_calendar(self):
        if not self.show_calendar:
            return

        return {
            'block': 'b-content',
            'elemMods': {'hidden': 'yes', 'calendar': 'yes'},
            'content': [
                {
                    'elem': 'item',
                    'elemMods': {'for-calendar': 'yes'},
                    'content': [{
                        'block': 'b-datepicker',
                        'js': True,
                        'content': [
                            {
                                'elem': 'month',
                                'content': [self.month_switcher(step) for step in xrange(12)]
                            },
                            {
                                'elem': 'calendar',
                                'content': [self.month_calendar(step) for step in xrange(12)]
                            },
                        ]
                    }]
                }
            ]
        }

    def head_content_extended(self):
        return []

    def metrika(self):
        return metrika.get_metrika(self.request)

    def meta_description(self):
        return tgettext(
            '''<title/>. На сервисе Яндекс.Расписания можно построить маршруты по всему миру с использованием самолётов, поездов, электричек и автобусов; посмотреть расписания и табло вокзалов и аэропортов по России и в странах СНГ, а также купить билет на самолёт, поезд или автобус.''',
            title=self.page_title()
        )

    def b_head_menu_item(self, url, menu_type, title, show_icon=False):
        content = []

        if show_icon:
            content.append({'block': 'b-icon'})

        content.append({'elem': 'text', 'content': title})

        return {
            'elem': 'item',
            'content': {
                'block': 'b-link',
                'mix': [{
                    'block': 'b-head-menu',
                    'elem': 'item',
                    'elemMods': {'type': menu_type}
                }],
                'url': url,
                'content': content
            }
        }

    def portal_menu_service_items(self):
        request = self.request

        items_data = (
            (LOCAL_PORTAL_SERVICES_URLS['images'][request.tld], gettext('Картинки'), 'images', True),
            (LOCAL_PORTAL_SERVICES_URLS['mail'][request.tld], gettext('Почта'), 'mail', True),
            (LOCAL_PORTAL_SERVICES_URLS['maps'][request.tld], gettext('Карты'), 'maps', True),
            (LOCAL_PORTAL_SERVICES_URLS['market'][request.tld], gettext('Маркет'), 'market', True),
            ('https://www.yandex.{domain}/all'.format(domain=request.tld), gettext('Все сервисы'), None, False)
        )

        return [self.b_head_menu_item(url, menu_type, title, show_icon)
                for url, title, menu_type, show_icon
                in items_data]

    def portal_menu_other_items(self):
        request = self.request
        yauser = request.yauser

        if yauser.is_authenticated():
            auth_item_title = gettext('Выход')

        else:
            auth_item_title = gettext('Вход')

        items_data = (
            (auth_url(request), auth_item_title),
            (request.change_region_url, xgettext('Регион: <city/>', city=request.client_city.L_title())),
            (feedback_url(request), gettext('Обратная связь'))
        )

        return [
            self.b_head_menu_item(url, 'options', title) for url, title in items_data
            if url is not None
        ]

    def b_head_menu(self):
        menu_items = [{
            'elem': 'title',
            'mix': [{'block': 'b-head-menu', 'elem': 'title'}],
            'content': gettext('Сервисы Яндекса')
        }]

        menu_items += self.portal_menu_service_items()

        menu_items += [{
            'elem': 'gap',
            'mix': [{'block': 'b-head-menu', 'elem': 'gap'}]
        }]

        menu_items += self.portal_menu_other_items()

        return {
            'block': 'b-head-menu',
            'mix': [{'block': 'i-services-request-controller', 'elem': 'menu'}],
            'content': menu_items
        }

    def e_favicon(self):
        e_favicons = get_favicons(self.request, settings.PROJECT_PATH)
        e_favicons['elem'] = 'icon'

        return e_favicons

    def fixed_content(self):
        return None

    def __json__(self):
        mods = None
        if hasattr(self, 'page_type'):
            mods = {'type': self.page_type}

        return [
            self.i_global(),
            {
                'block': 'b-page',
                'mods': mods,
                'title': self.page_title(),
                'head': [
                    self.e_favicon(),
                    {
                        'elem': 'css',
                        'url': self.asset_url('css'),
                        'ie': False
                    },
                    {'elem': 'js', 'url': '//yastatic.net/jquery/1.8.3/jquery.min.js'},
                    {'elem': 'js', 'url': self.asset_url('%s.pub.js' % get_language())},
                    {'elem': 'meta', 'attrs': {'name': 'msapplication-tap-highlight', 'content': 'no'}},
                    {'elem': 'meta', 'attrs': {'name': 'description', 'content': self.meta_description()}},
                    {'elem': 'link', 'attrs': {'rel': 'canonical', 'href': desktop_version_link(self.request)}}
                ] + [
                    {'elem': 'link', 'attrs': {'rel': 'alternate', 'href': url, 'hreflang': language}}
                    for url, language in self.alternative_links()
                ] + self.head_content_extended(),
                'content': [
                    {
                        'block': 'b-sideslide',
                        'mix': [{'block': 'i-ua', 'js': True}],
                        'content': [
                            {
                                'elem': 'content',
                                'content': [
                                    self.head_direct(),
                                    self.b_head(),
                                    {
                                        'block': 'b-content',
                                        'content': self.content() + [self.advert()]
                                    },
                                    self.content_calendar(),
                                    self.awaps(),
                                    self.b_footer(),
                                    self.share_popup(),
                                    self.metrika()
                                ]
                            },
                            {
                                'elem': 'menu',
                                'content': self.b_head_menu()
                            },
                            self.b_change_langs()
                        ]
                    },
                    self.fixed_content()
                ]
            }
        ]

    def b_change_langs(self):
        langs = self.request.supported_langs

        if len(langs) < 2:
            return

        current_lang = get_language()

        lang_names_map = dict(settings.LANGUAGES)

        items = [{'elem': 'title', 'content': gettext('Сменить язык')}]

        for lang in langs:
            mix = {'block': 'b-change-langs', 'elem': 'item'}
            attrs = {}

            if lang == current_lang:
                mix['mods'] = {'type': 'active'}
                url = None
            else:
                url = get_change_language_url(self.request, lang)
                attrs = metrika.reach_goal_on('click', 'lang-switch-click', params={'lang': lang})

            items.append({
                'block': 'b-link',
                'attrs': attrs,
                'url': url,
                'mix': mix,
                'content': gettext(unicode(lang_names_map[lang]), lang=lang)
            })

        return {
            'block': 'popup',
            'mods': {'type': 'modal', 'name': 'langs'},
            'underMods': {'type': 'paranja'},
            'content': [
                {
                    'elem': 'content',
                    'content': [
                        {
                            'block': 'b-change-langs',
                            'content': {
                                'elem': 'body',
                                'content': items
                            }
                        }
                    ]
                }
            ]
        }

    def share_popup(self):
        if not self.share:
            return

        number = self.context.thread.number
        title = self.context.thread.L_title()

        share_title = tgettext('Ссылка на рейс <number/>, <title/> на Яндекс.Расписаниях',
                               number=number,
                               title=title)

        subject = tgettext('Ссылка на рейс на Яндекс.Расписаниях')

        link = urlparse.urlunsplit(('https', settings.NATIONAL_RASP_DOMAINS[self.request.tld], self.request.path,
                                    '', ''))

        description = tgettext('Рейс <number/>, <title/> на Яндекс.Расписаниях: <link/>',
                               number=number,
                               title=title,
                               link=link)

        body = tgettext('Рейс <number/>, <title/>: <link/>',
                        number=number,
                        title=title,
                        link=link)

        if self.request.tld == 'com.tr':
            service_items = (
                ('mail', 'mail'),
                ('share-item', 'twitter'),
                ('share-item', 'facebook'),
                ('share-item', 'gplus'),
            )

        else:
            service_items = (
                ('mail', 'mail'),
                ('share-item', 'facebook'),
                ('share-item', 'twitter'),
                ('share-item', 'vkontakte'),
                ('share-item', 'lj'),
                ('share-item', 'gplus'),
            )

        return {
            'block': 'popup',
            'mods': {'type': 'modal', 'has-close': 'yes', 'name': 'likes'},
            'underMods': {'type': 'paranja'},
            'content': [
                {'elem': 'close'},
                {
                    'elem': 'content',
                    'content': [{
                        'block': 'likes',
                        'share': {
                            'title': share_title,
                            'description': description,
                            'image': '//yandex.st/lego/_/X31pO5JJJKEifJ7sfvuf3mGeD_8.png',
                            'subject': subject,
                            'body': body,
                            'url': link
                        },
                        'content': [{
                            'elem': elem,
                            'service': service
                        } for elem, service in service_items]
                    }]
                }
            ]
        }

    def awaps(self):
        return {
            'block': 'b-awaps',
            'content': '''\
<!-- Yandex.RTB R-I-239082-1 -->
<div id="yandex_rtb_R-I-239082-1"></div>
<script type="text/javascript">
    (function(w, d, n, s, t) {
        w[n] = w[n] || [];
        w[n].push(function() {
            Ya.Context.AdvManager.render({
                blockId: "R-I-239082-1",
                renderTo: "yandex_rtb_R-I-239082-1",
                horizontalAlign: false,
                async: true
            });
        });
        t = d.getElementsByTagName("script")[0];
        s = d.createElement("script");
        s.type = "text/javascript";
        s.src = "//an.yandex.ru/system/context.js";
        s.async = true;
        t.parentNode.insertBefore(s, t);
    })(this, this.document, "yandexContextAsyncCallbacks");
</script>'''
        }

    def head_direct(self):
        return {
            'block': 'b-awaps-in-header',
            'content': '''\
<!-- Yandex.RTB R-I-239082-2 -->
<div id="yandex_rtb_R-I-239082-2" style="height: 67px; border-bottom: 1px solid #d9d9d9;"></div>
<script type="text/javascript">
    (function(w, d, n, s, t) {
        w[n] = w[n] || [];
        w[n].push(function() {
            Ya.Context.AdvManager.render({
                blockId: "R-I-239082-2",
                renderTo: "yandex_rtb_R-I-239082-2",
                horizontalAlign: false,
                async: true
            });
        });
        t = d.getElementsByTagName("script")[0];
        s = d.createElement("script");
        s.type = "text/javascript";
        s.src = "//an.yandex.ru/system/context.js";
        s.async = true;
        t.parentNode.insertBefore(s, t);
    })(this, this.document, "yandexContextAsyncCallbacks");
</script>'''
        }

    def advert(self):
        request = self.request
        info = teaser_info(request)

        if not info:
            return

        return {
            'elem': 'item',
            'elemMods': {'type': 'tizer'},
            'content': [{
                'block': 'b-tizer',
                'url': info['url'],
                'src': info['imageUrl'],
                'width': info['imageWidth'],
                'content': info['text']
            }]
        }

    def i_global(self):
        return {
            'block': 'i-global',
            'params': {
                'rasp-url': settings.RASP_URL,
                'static-host': settings.MARKUP_URL,
                'touch-url': settings.TOUCH_URL,
                'url': settings.TOUCH_URL + self.request.get_full_path(),
                'suggests-domain': settings.SUGGESTS_DOMAIN,
                'client-city-geo-id': self.request.client_city._geo_id,
                'suggests-limit': settings.SUGGESTS_LIMIT,
                'lang': self.request.LANGUAGE_CODE,
                'national-version': self.request.NATIONAL_VERSION,
                'isLogin': self.request.yauser.is_authenticated()
            }
        }
