# -*- coding: utf-8 -*-

from __future__ import unicode_literals
from common.models.transport import TransportType

from common.utils.locations import composeurl
from common.xgettext.i18n import gettext

from .base import Base
from .form import SearchFormRenderer
from travel.rasp.touch.touch.core.helpers.transport_type import get_plural_transport_by
from travel.rasp.touch.touch.core.templates.blocks import metrika


class City(Base):
    page = 'index-favorites'
    show_calendar = True

    def list_item(self, item):
        content = self.b_list_item_content(item)
        return {
            'elem': 'item',
            'content': [
                {
                    'block': 'b-link',
                    'url': item['url'],
                    'mix': {'block': 'b-list', 'elem': 'item-text'},
                    'attrs': metrika.reach_goal_on(action='click',
                                                   target='MOVE-TO-TABLE',
                                                   params={'ttype': item['type']}),
                    'content': [
                        content,
                        {
                            'block': 'b-list',
                            'elem': 'icon'
                        }
                    ]
                },
            ]
        }

    def b_list_item_content(self, item):
        return [
            {
                'block': 'b-icon-carrier',
                'mods': {'type': item['type']}
            },
            item['title']
        ]

    def threads_search_widget(self):
        return {
            'block': 'b-wrapper',
            'attrs': metrika.reach_goal_on(action='submit',
                                           target='SEARCH-THREADS',
                                           other_attrs={
                                               'action': '/search-threads/',
                                               'method': 'get'
                                           }),
            'mods': {'type': 'search-input'},
            'content': {
                'block': 'b-search',
                'content': [
                    {
                        'elem': 'input',
                        'content': {
                            'block': 'input',
                            'mods': {'theme': 'city', 'size': 'm'},
                            'content': [
                                {
                                    'elem': 'label',
                                    'attrs': {'for': 'input-search-trip'},
                                    'content': gettext('Поиск рейса')
                                },
                                {
                                    'elem': 'control',
                                    'attrs': {
                                        'id': 'input-search-trips',
                                        'name': 'number',
                                        'placeholder': gettext('Рейс или поезд №')
                                    }
                                }
                            ]
                        }
                    },
                    {
                        'elem': 'button',
                        'content': {
                            'block': 'button',
                            'mods': {'theme': 'search', 'size': 'xl'},
                            'attrs': {'type': 'submit', 'role': 'submit'},
                            'content': gettext('Найти')
                        }
                    }
                ]
            }
        }

    def station_search_widget(self):
        return {
            'block': 'b-wrapper',
            'attrs': metrika.reach_goal_on(action='submit',
                                           target='SEARCH-STATIONS',
                                           other_attrs={
                                               'action': '/station-search/',
                                               'method': 'get'
                                           }),
            'mods': {'type': 'search-input'},
            'content': {
                'block': 'b-search',
                'content': [
                    {
                        'elem': 'input',
                        'content': {
                            'block': 'input',
                            'js': {
                                'dataprovider': {
                                    'name': 'form-dataprovider',
                                    'suggestsType': 'station'
                                },
                                'name': 'pointName'
                            },
                            'mods': {'suggest': 'yes', 'pick': 'yes', 'size': 'm', 'theme': 'city'},
                            'content': [
                                {
                                    'elem': 'label',
                                    'attrs': {'for': 'input-search-trip'},
                                    'content': gettext('Поиск станции')
                                },
                                {
                                    'elem': 'control',
                                    'attrs': {
                                        'id': 'input-search-trips',
                                        'placeholder': gettext('Название станции'),
                                        'name': 'pointName'
                                    }
                                },
                                {
                                    'elem': 'key',
                                    'tag': 'input',
                                    'attrs': {'name': 'pointId', 'value': '', 'type': 'hidden'}
                                }
                            ]
                        }
                    },
                    {
                        'elem': 'button',
                        'content': {
                            'block': 'button',
                            'mods': {'theme': 'search', 'size': 'xl'},
                            'attrs': {'type': 'submit', 'role': 'submit'},
                            'content': gettext('Найти')
                        }
                    }
                ]
            }
        }

    def change_city_widget(self):
        city = self.context.city

        return {
            'block': 'b-wrapper',
            'attrs': {'action': '/station-search/', 'method': 'get'},
            'mods': {'type': 'search-input'},
            'mix': {
                'block': 'b-change-city',
                'js': {
                    'fromPointId': city.point_key,
                    'fromPointName': city.L_title()
                }
            },
            'content': {
                'block': 'b-search',
                'content': [
                    {
                        'elem': 'input',
                        'content': {
                            'block': 'input',
                            'mods': {'type': 'city', 'suggest': 'yes', 'pick': 'yes', 'size': 'm', 'theme': 'city'},
                            'js': {
                                'dataprovider': {
                                    'name': 'form-dataprovider',
                                    'suggestsType': 'city'
                                },
                                'name': 'pointName',
                                'key-name': 'pointId'
                            },
                            'content': [
                                {
                                    'elem': 'control',
                                    'attrs': {'name': 'pointName', 'placeholder': city.L_title()}
                                },
                                {
                                    'elem': 'key',
                                    'tag': 'input',
                                    'attrs': {'name': 'pointId', 'value': '', 'type': 'hidden'}
                                }
                            ]
                        }
                    },
                    {
                        'elem': 'button',
                        'content': {
                            'block': 'button',
                            'mods': {'theme': 'search', 'size': 'xl'},
                            'attrs': {'type': 'submit', 'role': 'submit'},
                            'content': gettext('Сменить')
                        }
                    }
                ]
            }
        }

    def get_list_items(self):
        city = self.context.city
        stations = self.context.stations

        items = []

        if self.context.directions:
            items.append({
                'type': 'suburban',
                'title': gettext('Электрички'),
                'url': composeurl('directions', params={'city': city.id})
            })

        for ttype in ['plane', 'train', 'bus'] + TransportType.WATER_TTYPE_CODES:
            if ttype not in stations or not stations[ttype]['stations']:
                continue

            items.append({
                'type': ttype,
                'title': get_plural_transport_by(ttype),
                'url': composeurl('stations', kwargs={'ttype': ttype}, params={'city': city.id})
            })

        if self.context.aeroexpresses:
            items.append({
                'type': 'aeroex',
                'title': gettext('Аэроэкспрессы'),
                'url': composeurl('stations', kwargs={'ttype': 'plane'}, params={'city': city.id, 'aeroex': 1})
            })

        return items

    def transport_list(self):
        items = self.get_list_items()

        return {
            'elem': 'level',
            'elemMods': {'stage': 'second'},
            'mix': [{'block': 'b-list-transport'}],
            'content': [self.list_item(item) for item in items]
        }

    def content(self):

        return [
            {
                'elem': 'item',
                'content': SearchFormRenderer(self.context.form, show_cancel=False)
            },
            {
                'elem': 'item',
                'content': [
                    {
                        'block': 'b-list',
                        'content': [
                            {
                                'block': 'b-title',
                                'mods': {'type': 'list-on-favorites'},
                                'content': gettext('Табло и расписания')
                            },
                            {
                                'elem': 'level',
                                'elemMods': {'stage': 'first'},
                                'content': self.change_city_widget()
                            },
                            self.transport_list()
                        ]
                    }
                ]
            },
            {
                'elem': 'item',
                'elemMods': {'type': 'search-form'},
                'content': [{
                    'block': 'b-list',
                    'content': [{
                        'elem': 'level',
                        'elemMods': {'stage': 'second'},
                        'content': [{
                            'elem': 'item',
                            'elemMods': {'type': 'search-form'},
                            'content': [
                                self.threads_search_widget(),
                                self.station_search_widget()
                            ]
                        }]
                    }]
                }]
            }
        ]
