# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from cgi import escape
from common.utils.locations import composeurl

from common.xgettext.i18n import gettext, xgettext, dynamic_gettext
from travel.rasp.touch.touch.core.templates.blocks import metrika

from .utils import NBSP, cut_direction_full_title


class BInput(object):

    def __init__(self, form, name, suggests_type='common'):
        self.form = form
        self.name = name
        self.suggests_type = suggests_type

        self.field = form['%sName' % name]
        self.key_field = form['%sId' % name]

    def __json__(self):
        if hasattr(self.form, 'cleaned_data') and 'from' in self.form.cleaned_data:
            cleaned_data = self.form.cleaned_data
            value = cleaned_data[self.name].point.L_title()
            key_value = cleaned_data[self.name].point.point_key

        else:
            value = self.field.data or ''
            key_value = self.key_field.data or ''

        mods = {'suggest': 'yes', 'size': 'm', 'pick': 'yes', 'theme': 'destination'}

        if self.suggests_type == 'suburban' and self.name == 'to':
            mods['margin'] = 'bottom'

        return {
            'block': 'input',
            'mods': mods,
            'js': {
                'dataprovider': {
                    'name': 'form-dataprovider',
                    'field': self.name,
                    'suggestsType': self.suggests_type,
                    'ttype': self.form.ttype
                },
                'name': self.name
            },
            'content': [
                {
                    'elem': 'control',
                    'attrs': {
                        'placeholder': dynamic_gettext(self.field.label),
                        'autocorrect': 'off',
                        'name': self.field.name,
                        'value': value
                    }
                },
                {
                    'elem': 'key',
                    'tag': 'input',
                    'attrs': {
                        'name': self.key_field.name,
                        'value': key_value,
                        'type': 'hidden'
                    }
                }
            ]
        }


class WhenButton(object):

    def __init__(self, form, all_days=False):
        self.form = form
        self.field = form['when']
        self.all_days = all_days

    def __json__(self):
        mods = {'type': 'date'}
        attrs = {'type': 'button'}

        if self.all_days:
            attrs['style'] = 'background-color: #F3F2F0;'
            attrs['value'] = '' if self.form.is_now_today() else self.form.get_when_value()

        else:
            attrs['value'] = '' if self.form.is_now_today() else self.form.get_when()

        if not self.field.data and not self.all_days:
            mods['theme'] = 'destination'
            content = [dynamic_gettext(self.field.label)]

        else:
            content = [self.form.get_when()]

        attrs['data-delta'] = self.form.get_delta()

        content.append({'elem': 'button-icon'})
        return {
            'block': 'b-form',
            'elem': 'button',
            'elemMods': mods,
            'attrs': attrs,
            'content': [
                content,
                {'elem': 'button-icon'}
            ]
        }


class SearchFormRenderer(object):

    def __init__(self, form, base=None, hidden=False, show_cancel=True):
        self.form = form
        self.data = form.data
        self.hidden = hidden
        self.show_cancel = show_cancel
        self.base = base

    def __json__(self):
        js_params = {
            'fromName': self.data.get('fromName', ''),
            'fromId': self.data.get('fromId', ''),
            'toName': self.data.get('toName', ''),
            'toId': self.data.get('toId', ''),
            'when': '' if self.form.is_now_today() else self.form.get_when(),
            'delta': '' if self.form.is_now_today() else self.form.get_delta(),
            'defaultWhen': gettext('сегодня')
        }

        if hasattr(self.form, 'cleaned_data'):
            cleaned_data = self.form.cleaned_data

            if 'from' in cleaned_data:
                point_from = cleaned_data['from'].point
                point_to = cleaned_data['to'].point

                js_params.update({
                    'fromName': point_from.L_title(),
                    'fromId': point_from.point_key,
                    'toName': point_to.L_title(),
                    'toId': point_to.point_key
                })

        form_mods = {}

        if self.hidden:
            form_mods['hidden'] = 'yes'

        return {
            'block': 'b-form',
            'js': js_params,
            'mods': form_mods,
            'content': [
                BInput(self.form, 'from'),
                BInput(self.form, 'to'),
                WhenButton(self.form),
                {
                    'elem': 'switch',
                    'attrs': metrika.reach_goal_on(action='click',
                                                   target='RETURN-ROUTE',
                                                   other_attrs={'type': 'button'})
                },
                {
                    'elem': 'wrapper-button',
                    'content': {
                        'block': 'button',
                        'mods': {'theme': 'yellow', 'size': 'xl', 'type': 'search', '18': 'search'},
                        'attrs': {'type': 'submit'},
                        'content': gettext('Найти')
                    }
                },
                self.show_cancel and {
                    'elem': 'wrapper-button',
                    'content': {
                        'block': 'b-action-button',
                        'attrs': {'type': 'button'},
                        'mods': {'type': 'collapse-form', 'size': 'xl'},
                        'js': True,
                        'content': gettext('Отмена')
                    }
                }
            ]
        }

    def collapsed(self, mods=None, show_filter_button=False, board_route_js_params=True):
        if hasattr(self.form, 'cleaned_data') and 'from' in self.form.cleaned_data:
            cleaned_data = self.form.cleaned_data
            from_name = cleaned_data['fromName']
            to_name = cleaned_data['toName']

        else:
            from_name = self.form['fromName'].data
            to_name = self.form['toName'].data

        return {
            'block': 'b-board-route',
            'js': board_route_js_params,
            'mods': mods,
            'content': [
                show_filter_button and {
                    'elem': 'item',
                    'elemMods': {'position': 'left', 'type': 'filter'}
                },
                {
                    'elem': 'item',
                    'elemMods': {'position': show_filter_button and 'center-right' or 'single', 'type': 'route'},
                    'content': [
                        {
                            'elem': 'from-to',
                            'content': [
                                {'elem': 'text', 'content': from_name + ' &mdash;'},
                                ' ',
                                {'elem': 'text', 'content': to_name}
                            ]
                        },
                        {
                            'elem': 'text',
                            'elemMods': {'type': 'subtitle'},
                            'content': xgettext('отправление <date/>', date=self.form.get_when())
                        }
                    ]
                }
            ]
        }


class DirectionSearchFormRenderer(object):

    def __init__(self, form, base, hidden=False, show_cancel=True):
        self.form = form
        self.base = base
        self.mod = None
        self.hidden = hidden
        self.show_cancel = show_cancel

    def directions(self):
        current_dir = self.base.context.direction
        directions = self.base.context.directions

        selected = {'elem': 'selected', 'content': gettext('Все направления')}

        if current_dir:
            selected['content'] = cut_direction_full_title(current_dir)
            selected['value'] = current_dir.code

        return {
            'block': 'arrow-box',
            'mods': {'type': 'native-select', 'direction': 'list'},
            'value': current_dir and current_dir.code or None,
            'arrowOpt': [{
                'name': name,
                'value': code,
                'current': current_dir.code == code if current_dir else not i
            } for i, (code, name) in enumerate(directions)],
            'content': [
                selected
            ]
        }

    def station_widget(self, name, station_key):
        station_key = station_key or 'empty'
        stations = self.base.context.stations_list or []

        selected_station = next((s for s in stations if s['point_key'] == station_key), None)

        prefixes = {
            'from': gettext('От:'),
            'to': gettext('До:')
        }

        if stations:
            return {
                'block': 'arrow-box',
                'js': {'name': name},
                'mods': {'type': 'native-select', 'direction': 'stations', 'bold': 'yes'},
                'value': station_key,
                'arrowOpt': [{
                    'name': '',
                    'value': 'empty',
                    'disabled': True,
                    'current': station_key == 'empty'
                }] + [{
                    'name': station['title'],
                    'value': station['point_key'],
                    'current': station_key == station['point_key'],
                    'disabled': not bool(station['point_key'])
                } for i, station in enumerate(stations)],
                'content': [
                    prefixes[name],
                    NBSP,
                    {
                        'elem': 'selected',
                        'content': selected_station and selected_station['title'] or '',
                    }
                ]
            }

        return BInput(self.form, name, 'suburban')

    def mods(self):
        mods = {'type': 'suburban'}

        if self.hidden:
            mods['hidden'] = 'yes'

        return mods

    def all_days_button(self):
        mods = {'size': 'xl', 'pseudo': 'yes'}

        if self.form['all_days'].data:
            mods['checked'] = 'yes'

        return {
            'block': 'check-button',
            'value': 's',
            'mods': mods,
            'content': gettext('На все дни')
        }

    def when_button(self):
        return {
            'elem': 'wrapper-button',
            'content': WhenButton(self.form, int(self.form['all_days'].data))
        }

    def __json__(self):
        form_params = self.form.get_search_params(city_id=self.base.context.city.id, direction=self.base.context.direction)
        from_station_key = form_params['fromId'] or self.base.context.base_station_key or ''

        return {
            'block': 'b-form',
            'js': form_params,
            'mods': self.mods(),
            'content': [
                self.directions(),
                self.station_widget('from', from_station_key),
                {
                    'elem': 'switch',
                    'elemMods': {'type': 'electric-train'},
                    'attrs': metrika.reach_goal_on(action='click',
                                                   target='RETURN-ROUTE',
                                                   other_attrs={'type': 'button'})
                },
                self.station_widget('to', form_params['toId']),
                {
                    'block': 'b-search',
                    'content': [
                        {
                            'elem': 'input',
                            'content': self.when_button()
                        },
                        {
                            'elem': 'button',
                            'content': self.all_days_button()
                        }
                    ]
                },
                {
                    'elem': 'wrapper-button',
                    'content': {
                        'block': 'button',
                        'mods': {'theme': 'yellow', 'size': 'xl', 'type': 'search', '18': 'search'},
                        'attrs': {'type': 'submit'},
                        'content': gettext('Найти')
                    }
                },
                self.show_cancel and {
                    'elem': 'wrapper-button',
                    'content': {
                        'block': 'b-action-button',
                        'attrs': {'type': 'button'},
                        'mods': {'type': 'collapse-form', 'size': 'xl'},
                        'js': True,
                        'content': xgettext('Отмена')
                    }
                }
            ]
        }

    def collapsed(self):
        form_params = self.form.get_search_params()

        if self.form['all_days'].data:
            dep_content = gettext('расписание на все дни')

        else:
            dep_content = self.form.get_when()

        return {
            'block': 'b-board-route',
            'js': True,
            'mods': {'type': 'direction'},
            'content': [
                {
                    'elem': 'item',
                    'elemMods': {
                        'position': 'single',
                        'type': 'route'
                    },
                    'content': [
                        {
                            'elem': 'from-to',
                            'content': [
                                {'elem': 'text', 'content': form_params['fromName'] + ' &mdash;'},
                                ' ',
                                {'elem': 'text', 'content': form_params['toName']}
                            ]
                        },
                        {
                            'elem': 'text',
                            'elemMods': {'type': 'subtitle'},
                            'content': dep_content
                        }
                    ]
                },
                {
                    'elem': 'revert',
                    'content': {
                        'block': 'b-action-button',
                        'mix': {
                            'block': 'b-board-route',
                            'elem': 'revert-button'
                        },
                        'js': True,
                        'attrs': metrika.reach_goal_on(action='click',
                                                       target='RETURN-ROUTE'),
                        'mods': {
                            'type': 'return-route'
                        }
                    }
                }
            ]
        }
