# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from travel.rasp.library.python.common23.date.environment import get_locale
from common.utils.locations import langify
from common.xgettext.i18n import gettext, xgettext

from .base import Base
from travel.rasp.touch.touch.core.templates.blocks.companyIcon import CompanyIconModel, CompanyLogoTemplate
from travel.rasp.touch.touch.utils import get_station_url
from .utils import NBSP


class Template(Base):
    page = 'card-spot'

    def head_content_extended(self):
        return [{
            'elem': 'js',
            'url': '//api-maps.yandex.ru/2.0-stable/?load=package.standard&lang=%s' % get_locale()
        }]


class StationTemplate(Template):

    def address(self):
        station = self.context.station

        if not station.full_address:
            return

        if station.longitude and station.latitude:
            coords = [station.latitude, station.longitude]

        else:
            coords = None

        return {
            'block': 'b-card',
            'elem': 'item',
            'elemMods': {'type': 'address'},
            'mix': [
                {'block': 'i-map-toggler', 'js': True},
                {
                    'block': 'b-card',
                    'mods': {'switcher': 'maps'},
                    'js': {
                        'address': station.full_address,
                        'coords': coords
                    }
                }
            ],
            'content': [
                {
                    'block': 'b-card',
                    'elem': 'item-desc',
                    'content': [gettext('Адрес'), '<br/>']
                },
                station.full_address
            ]
        }

    def phones(self):
        blocks = []

        for phone in self.context.phones:
            numbers = [p.strip() for p in phone.phone.split(',')]
            phone_notes = [' ', phone.note] if len(self.context.phones) > 1 else []

            blocks.extend([{
                'block': 'b-link',
                'mix': [{'block': 'b-card', 'elem': 'item', 'elemMods': {'type': 'phone'}}],
                'url': 'tel:' + number,
                'content': [
                    {
                        'block': 'b-card',
                        'elem': 'item-desc',
                        'content': [gettext('Телефон')] + phone_notes + [' ', '<br/>']
                    },
                    number
                ]
            } for number in numbers])

        return blocks

    def title(self):
        station = self.context.station
        tcode = station.t_type.code

        if tcode == 'plane':
            if station.iata:
                title = xgettext('Аэропорт <title/><iata-code/>',
                                 title=station.L_title(),
                                 iata_code=lambda: {'elem': 'code', 'content': station.iata})

            else:
                title = xgettext('Аэропорт <title/>', title=station.L_title())

            return title, gettext('Табло аэропорта')

        if tcode == 'train':
            return xgettext('Вокзал <title/>', title=station.L_title()), gettext('Расписание')

        return xgettext('Станция <title/>', title=station.L_title()), gettext('Расписание')

    def map(self):
        station = self.context.station

        if not station.full_address:
            return

        return {
            'block': 'b-map',
            'js': True,
            'mods': {'theme': 'top-fade', 'visibility': 'hidden'},
            'attrs': {'id': 'map', 'style': 'width: 100%; height: 250px;'}
        }

    def weather_elem(self):
        weather = self.context.weather

        if not weather:
            return

        icon_url = None

        if 'image-v3' in weather:
            icon_url = '//yandex.st/weather/i/icons/22x22/%s.png' % weather['image-v3']

        return {
            'elem': 'local',
            'elemMods': {'type': 'weather'},
            'content': [
                icon_url and {
                    'elem': 'local-weather-ico',
                    'tag': 'img',
                    'attrs': {'src': icon_url, 'width': '16px'}
                },
                {
                    'elem': 'local',
                    'elemMods': {'type': 'weather-text'},
                    'content': weather.get('temperature')
                }
            ]
        }

    def local_elem(self):
        station = self.context.station
        now = self.request.now.replace(tzinfo=None)
        local_now = station.get_local_datetime(now)

        return {
            'elem': 'local',
            'content': [
                {
                    'elem': 'local',
                    'elemMods': {'type': 'time'},
                    'content': [
                        gettext('местное время'),
                        NBSP,
                        {
                            'elem': 'time',
                            'content': local_now.strftime('%H:%M')
                        }
                    ]
                },
                self.weather_elem()
            ]
        }

    def content(self):
        station = self.context.station

        title, button_title = self.title()

        contacts = []

        if station.site_url:
            contacts = [{
                'block': 'b-link',
                'mix': [{'block': 'b-card', 'elem': 'item', 'elemMods': {'type': 'site'}}],
                'target': '_blank',
                'url': station.site_url,
                'content': [
                    {
                        'block': 'b-card',
                        'elem': 'item-desc',
                        'content': [
                            gettext('Сайт'),
                            '<br/>'
                        ]
                    },
                    station.site_url
                ]
            }]

        contacts += self.phones()
        contacts += [self.address()]

        return [{
            'elem': 'item',
            'elemMods': {'type': 'card'},
            'content': [
                {
                    'block': 'b-card',
                    'content': [
                        {
                            'elem': 'head',
                            'content': [
                                {
                                    'elem': 'title',
                                    'content': title
                                },
                                self.local_elem(),
                                {
                                    'block': 'b-action-button',
                                    'mods': {'type': 'card'},
                                    'url': langify(get_station_url(kwargs={'sid': station.id})),
                                    'content': button_title
                                },
                                {
                                    'elem': 'description',
                                    'content': station.L_how_to_get_to_city()
                                }
                            ]
                        },
                        {
                            'elem': 'contacts',
                            'content': contacts
                        },
                    ]
                },
                {
                    'elem': 'tail',
                    'elemMods': {'visibility': 'hidden'}
                }
            ]
        }, self.map()]


class CompanyTemplate(Template):

    def codes(self):
        company = self.context.company

        iata_codes = []

        if company.sirena_id:
            iata_codes.append(company.sirena_id)

        if company.iata:
            iata_codes.append(company.iata)

        if not iata_codes:
            return

        return {
            'block': 'b-card',
            'elem': 'item',
            'content': [
                {
                    'block': 'b-card',
                    'elem': 'item-desc',
                    'content': [gettext('Код IATA / &#0171;Сирена&#0187;'), '<br/>']
                },
                ' / '.join(iata_codes),
            ]
        }

    def contact_info(self):
        company = self.context.company

        if not company.contact_info:
            return

        return {
            'block': 'b-card',
            'elem': 'item',
            'content': [
                {
                    'block': 'b-card',
                    'elem': 'item-desc',
                    'content': [gettext('Контактная информация'), '<br/>']
                },
                company.contact_info
            ]
        }

    def country(self):
        company = self.context.company

        if not company.country:
            return

        return {
            'elem': 'country',
            'content': company.country.L_title()
        }

    def logo(self):
        company = self.context.company

        if not company.logo:
            return

        return {
            'elem': 'logo',
            'content': CompanyLogoTemplate(CompanyIconModel.get_icon_from_company(company))
        }

    def company_url(self):
        company = self.context.company

        if not company.url:
            return

        return {
            'block': 'b-link',
            'attrs': {'target': '_blank'},
            'mix': [{'block': 'b-card', 'elem': 'item', 'elemMods': {'type': 'site'}}],
            'content': [
                {
                    'block': 'b-card',
                    'elem': 'item-desc',
                    'content': [gettext('Сайт'), '<br/>']
                },
                company.url
            ],
            'url': company.url
        }

    def email(self):
        company = self.context.company

        if not company.email:
            return

        return {
            'block': 'b-link',
            'mix': [{'block': 'b-card', 'elem': 'item', 'elemMods': {'type': 'email'}}],
            'url': 'mailto:' + company.email,
            'content': [
                {
                    'block': 'b-card',
                    'elem': 'item-desc',
                    'content': [gettext('E-mail'), '<br/>']
                },
                company.email
            ]
        }

    def phone(self):
        company = self.context.company

        if not company.phone:
            return

        return {
            'block': 'b-link',
            'mix': [{'block': 'b-card', 'elem': 'item', 'elemMods': {'type': 'phone'}}],
            'url': 'tel:' + company.phone,
            'content': [
                {
                    'block': 'b-card',
                    'elem': 'item-desc',
                    'content': [gettext('Телефон'), '<br/>']
                },
                company.phone
            ]
        }

    def address(self):
        company = self.context.company

        if not company.address:
            return

        return {
            'block': 'b-card',
            'elem': 'item',
            'elemMods': {'type': 'address'},
            'js': {'address': company.address},
            'mix': [{
                'block': 'b-card',
                'mods': {'switcher': 'maps'},
                'js': {
                    'address': company.address,
                }
            }],
            'content': [
                {
                    'block': 'b-card',
                    'elem': 'item-desc',
                    'content': [gettext('Адрес'), '<br/>']
                },
                company.address
            ]
        }

    def map(self):
        company = self.context.company

        if not company.address:
            return

        return {
            'block': 'b-map',
            'js': True,
            'mods': {'theme': 'top-fade', 'visibility': 'hidden'},
            'attrs': {'id': 'map', 'style': 'width: 100%; height: 250px;'}
        }

    def content(self):
        company = self.context.company

        return [
            {
                'elem': 'item',
                'elemMods': {'type': 'card'},
                'content': [
                    {
                        'block': 'b-card',
                        'content': [
                            {
                                'elem': 'head',
                                'content': [
                                    {
                                        'elem': 'title',
                                        'content': company.L_title()
                                    },
                                    {
                                        'elem': 'info',
                                        'content': [
                                            self.country(),
                                            self.logo()
                                        ]
                                    },
                                ]
                            },
                            {
                                'elem': 'contacts',
                                'content': [
                                    self.codes(),
                                    self.contact_info(),
                                    self.company_url(),
                                    self.email(),
                                    self.phone(),
                                    self.address(),
                                ]
                            }
                        ]
                    },
                    {
                        'elem': 'tail',
                        'elemMods': {'visibility': 'hidden'}
                    }
                ]
            },
            self.map()
        ]
