# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from datetime import timedelta, time, datetime, date
import uuid

from django.core.urlresolvers import reverse
import math
from common.models.transport import TransportType

from travel.rasp.library.python.common23.date import environment
from common.utils.blablacar_utils import BLABLACAR_ALL_DAYS_DATE, is_valid_blablacar_direction
from common.utils.date import uni_strftime, NBSP
from common.utils.locations import composeurl, langify
from common.xgettext.i18n import gettext, xgettext, ngettext, dynamic_gettext, tngettext
from common.models.partner import Partner
from travel.rasp.touch.touch.core.helpers.blablacar import pop_blablacar_row
from travel.rasp.touch.touch.core.templates.blocks.buy_avia_ticket import AviaDisclaimer

from travel.rasp.touch.touch.core.templates.blocks.companyIcon import CompanyIconModel, CompanyIconTemplate
from ..base import Base
from ..blocks import PreloaderBlock, ErrorDisclaimerBlock, DetailsAboutLocalTime
from ..form import SearchFormRenderer
from travel.rasp.touch.touch.core.templates.elements.lineInfo import build_segment_line_info
from travel.rasp.touch.touch.core.templates.blocks.tariffPrice import TariffPrice
from travel.rasp.touch.touch.core.views.utils import get_transport_code_from, get_attr_by_path
from ..utils import wrap_with_link_item, wrap_with_span_item, human_duration, currency


ROUTE_PICTURE_TYPES = {
    'suburban': 'electric-train',
    'train': 'train',
    'aeroex': 'train',
    'bus': 'bus',
    'sea': 'water',
    'river': 'water',
    'water': 'water',
    'helicopter': 'avia',
    'plane': 'avia',
}

BLABLACAR_POSITION = 2
BLABLACAR_AWAPS_SHOW_URL = None
BLABLACAR_AWAPS_CLICK_URL = None


class TimeWithDateIfChangedFormatter(object):

    def __init__(self, base):
        self.base = base
        self.prev_date = None

    def __call__(self, dt, hidden=False):
        if dt.date() != self.prev_date:
            self.prev_date = dt.date()
            fmt = '%d' + NBSP + '%b %H:%M'

            return uni_strftime(fmt, dt)

        return uni_strftime('%H:%M', dt)


class Template(Base):
    page = 'search-result'
    show_calendar = True

    def __init__(self, request, view_context):
        super(Template, self).__init__(request, view_context)

        if not self.context.no_airports_error and self.context.groups and self.context.ajax_tariffs_info:
            self.is_ajax = True

        else:
            self.is_ajax = False

    @classmethod
    def ttype_title(cls, ttype, n):
        if ttype == 'bus':
            return ngettext(n, '%d Автобус', '%d Автобуса', '%d Автобусов') % n

        if ttype == 'train':
            return ngettext(n, '%d Поезд', '%d Поезда', '%d Поездов') % n

        if ttype == 'plane':
            return ngettext(n, '%d Самолёт', '%d Самолёта', '%d Самолётов') % n

        if ttype in TransportType.WATER_TTYPE_CODES:
            return ngettext(n, '%d Теплоход', '%d Теплохода', '%d Теплоходов') % n

        if ttype == 'suburban':
            return ngettext(n, '%d Электричка', '%d Электрички', '%d Электричек') % n

        if ttype == 'helicopter':
            return ngettext(n, '%d Вертолёт', '%d Вертолёта', '%d Вертолётов') % n

        if ttype == 'aeroex':
            return ngettext(n, '%d Аэроэкспресс', '%d Аэроэкспресса', '%d Аэроэкспрессов') % n

    def line(self, ttype, data):
        cnt = data['cnt']
        duration = data['duration']
        form = self.context.form
        search_params = form.get_search_params(is_raw=True)

        if ttype == 'aeroex':
            search_params.update({'aeroex': 'y'})
            search_type = 'suburban'

        elif ttype in TransportType.WATER_TTYPE_CODES:
            search_type = 'water'

        else:
            search_type = ttype

        search_params = dict((k, v) for k, v in search_params.iteritems() if v)

        url = composeurl('search', kwargs={'search_type': search_type}, params=search_params)

        price = currency(data['min_tariff'], self.context.currency_info, cnt > 1)

        if cnt > 1:
            duration_text = xgettext('от <duration/>', duration=human_duration(duration))

        else:
            duration_text = human_duration(duration)

        return {
            'elem': 'line',
            'js': {
                'count': cnt,
                'ttype': ttype,
                'min-duration': {'days': duration.days, 'seconds': duration.seconds},
                'min-tariff': data['min_tariff']
            },
            'content': [
                {
                    'elem': 'item',
                    'elemMods': {'column': 'left'},
                    'content': [
                        {
                            'block': 'b-link',
                            'js': True,
                            'url': url,
                            'mix': [{'block': 'b-transport', 'elem': 'title'}],
                            'content': [
                                {
                                    'block': 'b-icon-carrier',
                                    'mods': {'type': ttype}
                                },
                                self.ttype_title(ttype, cnt)
                            ]
                        }
                    ]
                },
                {
                    'elem': 'item',
                    'elemMods': {'column': 'middle'},
                    'content': {
                        'elem': 'time-in-path',
                        'content': duration_text
                    }
                },
                {
                    'elem': 'item',
                    'elemMods': {'column': 'right'},
                    'content': {
                        'elem': 'price',
                        'content': price
                    }
                },
            ]
        }

    def timetable(self):
        point_from = self.context.point_from
        point_to = self.context.point_to
        form = self.context.form

        if self.context.no_airports_error:
            return {
                'block': 'b-transport',
                'content': ErrorDisclaimerBlock(xgettext(u'Аэропорты в стране <country-title/> отсутствуют, или до них нет известных нам рейсов. Пожалуйста, выберите другой пункт отправления.',
                                                         country_title=self.context.country.L_title()))
            }

        not_found_message = xgettext('К сожалению, у Яндекс.Расписаний нет данных о рейсах <point_from/> — <point_to/> на дату <when/>.',
                                     point_from=point_from.L_title,
                                     point_to=point_to.L_title,
                                     when=form.get_when())

        groups = self.context.groups
        ajax_info = self.context.ajax_tariffs_info

        if self.is_ajax:
            mods = {'ajax': 'yes'}

        else:
            mods = None

        if groups:
            content = [self.line(ttype, data) for ttype, data in groups.iteritems()]

        else:
            content = [ErrorDisclaimerBlock(not_found_message)]

        return [
            #TODO
            False and self.context.grouped_variants.extended and {
                'block': 'error-message',
                'content': self.context.grouped_variants.extended_message
            },
            {
                'block': 'b-transport',

                'mods': mods,
                'mix': {'block': 'b-list-transport'},
                'js': {
                    'ajax_info': ajax_info,
                    'when': form['when'].data.strftime('%Y-%m-%d'),
                    'update-url': langify(reverse('main_tariffs')),
                    'not_found_message': not_found_message
                },
                'content': content
            }
        ]

    def extended_notice(self):
        if self.context.groups and self.context.extended:
            message = xgettext('Прямых рейсов <request-direction/> не найдено. Предлагаем вам результат поиска <display-direction/>.',
                               request_direction="%s — %s" % (self.context.requested_from.L_title_with_prefix(), self.context.requested_to.L_title_with_prefix()),
                               display_direction="%s — %s" % (self.context.point_from.L_title_with_prefix(), self.context.point_to.L_title_with_prefix()),)

            return ErrorDisclaimerBlock(message)

    def content(self):
        if self.context.no_airports_error:
            form_hidden = False
            show_cancel = False

        else:
            form_hidden = True
            show_cancel = True

        form_renderer = SearchFormRenderer(self.context.form, hidden=form_hidden, show_cancel=show_cancel)

        return [
            self.alarm(),
            {
                'elem': 'item',
                'content': [
                    not self.context.no_airports_error and form_renderer.collapsed(),
                    form_renderer,
                    self.extended_notice(),
                    self.timetable(),
                    self.is_ajax and PreloaderBlock()
                ]
            },
        ]


class TimetableBlock(object):
    mix = None

    def __init__(self, base, **kwargs):
        self.base = base

        self.context = base.context

        self.show_time_icons = kwargs.get('show_time_icons')

        self.show_company_icon = kwargs.get('show_company_icon')

        data = base.context.data

        self.data = data

        segments = self.context.segments

        self.segments = segments

        self.form = self.context.form

        if self.form:
            self.ttype = self.form.ttype
            self.when = self.form.data['when']

        else:
            self.ttype = self.context.ttype
            self.when = self.context.when

        if self.form and not self.context.extended:
            self.point_from = self.form.cleaned_data['from'].point
            self.point_to = self.form.cleaned_data['to'].point

        else:
            self.point_from = self.context.point_from
            self.point_to = self.context.point_to

        if 'hide_gones' in kwargs:
            self.hide_gones = kwargs['hide_gones']

        else:
            self.hide_gones = True

        self.search_type = self.ttype

        if 'show_not_found_message' in kwargs:
            self.show_not_found_message = kwargs['show_not_found_message']

        else:
            self.show_not_found_message = True

        if 'ajax' in kwargs:
            self.is_ajax = kwargs['ajax']
        else:
            self.is_ajax = True

        if 'show_tariffs_column' in kwargs:
            self.show_tariffs_column = kwargs['show_tariffs_column']

        else:
            self.show_tariffs_column = True

        self.show_prices = kwargs.get('show_prices',  True)

        if 'mods' in kwargs:
            self.mods = kwargs['mods']

        else:
            self.mods = None

        if 'js_params' in kwargs:
            self.js_params = kwargs['js_params']

        else:
            self.js_params = {}

        if 'show_time_left' in kwargs:
            self.show_time_left = kwargs['show_time_left']

        else:
            self.show_time_left = True

        if 'show_date_if_changed' in kwargs:
            self.show_date_if_changed = kwargs['show_date_if_changed']

        else:
            self.show_date_if_changed = True

        self.is_aeroex = 'aeroex' in self.context.request.GET

        self.show_stops = kwargs.get('show_stops')

        self.show_days = kwargs.get('show_days')

        self.has_gone = any([s.gone for s in segments])

        self.has_actual = any([not s.gone for s in segments])

        self.show_gone = self.has_gone and not self.has_actual

        if self.show_date_if_changed:
            self.time_formatter = TimeWithDateIfChangedFormatter(base)
            self.arrival_formatter = TimeWithDateIfChangedFormatter(base)
            self.departure_formatter = TimeWithDateIfChangedFormatter(base)

        else:
            self.time_formatter = self.arrival_formatter = self.departure_formatter = lambda dt: uni_strftime('%H:%M', dt)

    def picture(self, segment):
        if self.show_company_icon:
            return CompanyIconTemplate(CompanyIconModel.get_icon_from_segment(segment))

        from_point = self.point_from
        to_point = self.point_to
        end_stations_ids = self.context.end_stations_ids

        segment_line_mod = None

        if not getattr(segment, 'is_extra', False):
            thread = segment.thread

            departure_station_ids = end_stations_ids[thread]['departure']
            arrival_station_ids = end_stations_ids[thread]['arrival']

            if from_point.id not in departure_station_ids and to_point.id not in arrival_station_ids:
                segment_line_mod = 'top-bottom'

            elif from_point.id not in departure_station_ids:
                segment_line_mod = 'top'

            elif to_point.id not in arrival_station_ids:
                segment_line_mod = 'bottom'

        t_code = get_transport_code_from(segment)
        type_color = ROUTE_PICTURE_TYPES.get(t_code, None)

        return {
            'block': 'b-route',
            'content': {
                'elem': 'picture',
                'elemMods': {'line': segment_line_mod, 'type': type_color}
            }
        }

    def has_eticket(self, segment):
        tariffs_info = segment.display_info.get('tariffs_info')

        if not tariffs_info:
            return False

        return bool(tariffs_info.et_marker)

    def in_path(self, segment):
        text = [human_duration(segment.duration)]

        if self.show_stops:
            text += ['<br/>'] + xgettext('остановки: <stops/>', stops=segment.L_stops)

        if self.show_days:
            shift = (segment.departure.date() - segment.start_date).days
            text += ['<br/>', segment.thread.L_days_text(shift, next_plan=getattr(segment, 'next_plan', None))]

        return {
            'elem': 'time',
            'elemMods': {'type': 'in-path'},
            'js': True,
            'content': text
        }

    def line(self, segment):
        national_version = self.context.request.NATIONAL_VERSION
        mix = [{'block': 'i-pressed-controller'}]

        hide_line = self.hide_gones and segment.gone and not self.show_gone

        if self.base.show_only_aeroexpress and not get_attr_by_path(segment, 'thread.is_aeroexpress', False):
            hide_line = True

        if hide_line:
            mix.append({'block': 'i-hidden'})

        mods = {'state': 'active'}

        if self.hide_gones and segment.gone:
            mods['gone'] = 'yes'

        if getattr(segment, 'is_extra', False):
            mods['type'] = 'pop'

        return {
            'block': 'b-routers',
            'elem': 'line',
            'elemMods': mods,
            'mix': mix,
            'hidden': hide_line,
            'js': self.line_js_params(segment),
            'content': [
                {
                    'elem': 'item',
                    'elemMods': {'type': 'flag'},
                    'content': None
                },
                {
                    'elem': 'item',
                    'content': [
                        {
                            'elem': 'time',
                            'elemMods': {'type': 'departure', 'with-icon': self.show_time_icons and 'yes'},
                            'content': self.departure_formatter(segment.departure)
                        },
                        {
                            'elem': 'time',
                            'elemMods': {'type': 'arrival', 'with-icon': self.show_time_icons and 'yes'},
                            'content': self.arrival_formatter(segment.arrival)
                        }
                    ]
                },
                {
                    'elem': 'item',
                    'elemMods': {'type': 'route-picture'},
                    'content': self.picture(segment)
                },
                {
                    'elem': 'item',
                    'elemMods': {'type': 'route'},
                    'content': [
                        build_segment_line_info(segment=segment, national_version=national_version),
                        self.thread_title_link(segment),
                        self.in_path(segment)
                    ]
                },
                self.tariffs_column(segment),
            ]
        }

    def blablacar_line(self, segment=None):
        if not is_valid_blablacar_direction(self.context.point_from, self.context.point_to):
            return None

        blablacar_date = self.when or BLABLACAR_ALL_DAYS_DATE

        blablacar_request_key = 'blablacar_%s' % blablacar_date.isoformat() if isinstance(blablacar_date, date) else None
        if not (segment or blablacar_request_key and self.context.ajax_tariffs_info and blablacar_request_key in self.context.ajax_tariffs_info.request_keys):
            return None
        blablacar_content = None

        if segment:
            # округление продолжительности пути до 10 минут в большую сторону
            duration_in_sec = math.trunc(math.ceil(segment.duration.total_seconds() / 600)) * 600
            duration = timedelta(seconds = duration_in_sec)

            data = {
                'route': segment.L_title().replace(u' - ', u' — '),
                'time': '~%s' % human_duration(duration),
                'count': tngettext(
                    segment.number_of_variants,
                    '<n/> предложение',
                    '<n/> предложения',
                    '<n/> предложений'
                ) if hasattr(segment, 'number_of_variants') else None,
            }

            tariffs = segment.display_info.tariffs_info
            if tariffs:
                data['url'] = getattr(tariffs, 'm_url', None)

                if tariffs.places:
                    place = min(tariffs.places, key=lambda x: x.tariff)

                    data['tariff'] = currency(place.tariff,  self.context.currency_info, True)

        else:
            data = None
            blablacar_content = [
                {
                    'block': 'spin2',
                    'js': True,
                    'mods': { 'size': 's', 'progress': 'yes' }
                },
                {
                    'elem': 'text',
                    'content': xgettext(u'Ищем предложения')
                }
            ]

        return {
            'block': 'b-routers',
            'elem': 'line',
            'elemMods': { 'have-flag': 'yes', 'type': 'blablacar' },
            'content': [
                {
                    'elem': 'item',
                    'elemMods': { 'type': 'flag' },
                    'content': {
                        'elem': 'flag',
                        'elemMods': { 'type': 'blablacar' },
                        'content': {
                            'elem': 'ad',
                            'content': xgettext(u'реклама')
                        }
                    }
                },
                {
                    'elem': 'item',
                    'attrs': {
                        'colspan': 4
                    },
                    'content': {
                        'block': 'b-blablacar',
                        'mods': {
                            'loading': 'success' if data else 'yes'
                        },
                        'js': {
                            'data': data,
                            'awapsClickUrl': BLABLACAR_AWAPS_CLICK_URL,
                            'awapsShowUrl': BLABLACAR_AWAPS_SHOW_URL,
                        },
                        'content': blablacar_content
                    }
                }
            ]
        }

    def thread_title_link(self, segment):
        thread_link = None

        is_extra = getattr(segment, 'is_extra', False)

        if not is_extra and segment.thread.uid:
            if self.search_type == 'plane':
                thread = segment.thread
                dt = datetime.combine(self.form.data['when'], time(0, 0))
                departure = thread.pytz.localize(dt)

            else:
                departure = segment.start_date

            params = {
                  'departure': departure.strftime('%Y-%m-%d'),
                  'date': segment.departure.date(),
                  'station_to': segment.station_to.id,
                  'station_from': segment.station_from.id,
                  'point_from': self.point_from.point_key,
                  'point_to': self.point_to.point_key,
                  'number': segment.number,
                  't_type': segment.t_type.code,
                  'thread': segment.thread.uid
            }

            thread_link = composeurl('thread', kwargs={'uid': segment.thread.uid}, params=params)

        if not is_extra:
            wrapper = wrap_with_link_item

        else:
            wrapper = wrap_with_span_item
        return {
            'block': 'b-link',
            'js': {
                'url': thread_link
            },
            'mix': [{'block': 'b-route'}],
            'url': thread_link,
            'content': wrapper(segment.get_popular_title())
        }

    def get_day_time(self, dt):
        t = dt.time()

        if t is None:
            return

        if t < time(6, 0):
            return 'night'

        if t < time(12, 0):
            return 'morning'

        if t < time(18, 0):
            return 'day'

        return 'evening'

    def min_price(self, segment):
        tariffs_info = segment.display_info.get('tariffs_info')

        if not tariffs_info or segment.gone:
            return

        places = tariffs_info.places

        if not places:
            return 0

        price = min([place.tariff for place in places])
        min_price = price.base_value or price.value

        return min_price or 0

    def format_filter_station_name(self, station):
        return station.L_title()

    def line_js_params(self, segment):
        loc_now = segment.station_from.get_local_datetime(environment.now())
        time_to_dep = segment.departure - loc_now

        time_to_departure = None

        if time_to_dep < timedelta(minutes=1):
            time_to_departure = gettext('отправляется')

        elif time_to_dep <= timedelta(days=3):
            time_to_departure = xgettext(u'через <time/>', time=human_duration(time_to_dep))

        minutes_in_path = segment.duration.days * 24 * 60 + segment.duration.seconds / 60

        day_times = {
            'morning': gettext('утро'),
            'day': gettext('день'),
            'evening': gettext('вечер'),
            'night': gettext('ночь')
        }

        # генерим id для сортировки сегментов на клиенте
        if getattr(segment, 'thread', None):
            segment_id = segment.thread.uid
        else:
            segment_id = str(uuid.uuid4())

        price = self.min_price(segment)
        return {
            'id': segment_id,
            'key': self.is_ajax and segment.info and segment.info.route_key,
            'type': get_transport_code_from(segment),
            'is_extra': getattr(segment, 'is_extra', False),
            'departure': segment.departure.strftime('%d %B %Y %H:%M'),
            'arrival': segment.arrival.strftime('%d %B %Y %H:%M'),
            'travel_time': minutes_in_path,
            'price': self.min_price(segment),
            'time_to_departure': time_to_departure,
            'station-from': [{
                'name': self.format_filter_station_name(segment.station_from),
                'value': segment.station_from.id
            }],
            'station-to': [{
                'name': self.format_filter_station_name(segment.station_to),
                'value': segment.station_to.id
            }],
            'company': segment.thread and [{
                'value': segment.thread.company and segment.thread.company.id,
                'name': segment.thread.company and segment.thread.company.L_title()
            }] or None,
            'tickets': [{
                'value': ticket,
                'name': ticket
            } for ticket in []],
            'eticket': [{
                'value': self.has_eticket(segment),
                'name': ''
            }],
            'is-express': [{
                'value': 'yes' if segment.thread and segment.thread.is_express else 'no',
                'name': ''
            }],
            'is-aeroex': [{
                'value': 'yes' if segment.thread and segment.thread.is_aeroexpress else 'no',
                'name': ''
            }],
            'has-price': [{
                'value': 'yes' if price > 0 else 'no',
                'name': ''
            }],
            'departure-time': [{
                'value': self.get_day_time(segment.departure),
                'name': day_times.get(self.get_day_time(segment.departure))
            }],
            'arrival-time': [{
                'value': self.get_day_time(segment.arrival),
                'name': day_times.get(self.get_day_time(segment.arrival))
            }],
            'gone': segment.gone,
        }

    def tariffs_column(self, segment):
        if not self.show_tariffs_column:
            return

        if self.hide_gones and segment.gone:
            return {
                'elem': 'item',
                'elemMods': {'type': 'last-column'},
                'content': gettext('ушёл')
            }

        if not self.show_prices:
            return {
                'elem': 'item',
                'elemMods': {'type': 'last-column'},
                'content': None
            }

        tariffs_info = segment.display_info.get('tariffs_info')

        if not tariffs_info:
            return {'elem': 'item', 'elemMods': {'type': 'last-column'}}

        if self.search_type == 'plane':
            places = tariffs_info.places

            content = []

            if places:
                place = min(places, key=lambda x: x.tariff)
                content = [
                    {
                        'elem': 'price',
                        'content': currency(place.tariff, self.context.currency_info, getattr(place, 'min', False))
                    },
                    {
                        'elem': 'button',

                        'content': {
                            'block': 'button',
                            'mods': {
                                'theme': 'yellow',
                                'size': 's'
                            },
                            'attrs': {
                                'target': '_blank'
                            },
                            'url': place.deep_link,
                            'content': xgettext('Купить')
                        }
                    }
                ]

        else:
            content = [TariffPrice(place,
                                   self.context.currency_info,
                                   'train',
                                   national_version=self.base.request.NATIONAL_VERSION)

                       for place in tariffs_info.places]

        if self.search_type in ('bus', 'train', 'plane') and not tariffs_info.places and not segment.gone:
            show_spin = True

        else:
            show_spin = False

        return {
            'elem': 'item',
            'elemMods': {'type': 'last-column'},
            'content': show_spin and {
                'block': 'b-spin',
                'mods': {
                    'size': '10',
                    'theme': 'white-10',
                    'progress': 'yes',
                    'request-key': self.is_ajax and segment.info.route_key
                }
            } or content
        }

    def not_found_message(self):
        if not self.show_not_found_message:
            return

        return xgettext('''К сожалению, у Яндекс.Расписаний нет данных о рейсах <point_from/> — <point_to/> на дату <when/>.''',
                        point_from=self.form['fromName'].data,
                        point_to=self.form['toName'].data,
                        when=self.form.get_when())

    def suburban_tariff(self):
        if self.search_type != 'suburban':
            return

        segment = self.segments[0]

        tariffs_info = segment.display_info.get('tariffs_info')

        if not tariffs_info:
            return

        price = currency(min(p.tariff for p in tariffs_info.places), self.context.currency_info)

        if self.is_aeroex:
            content = xgettext('<price/> за билет на экспресс',
                               price=lambda: {'elem': 'price', 'elemMods': {'type': 'electric-train'}, 'content': price})

        else:
            content = xgettext('<price/> за обычный билет',
                               price=lambda: {'elem': 'price', 'elemMods': {'type': 'electric-train'}, 'content': price})

        return {
            'block': 'b-routers',
            'elem': 'price-row',
            'content': content
        }

    @classmethod
    def ajax_lines(cls, request, context, segments, search_type):
        base = TimetableTemplate(request, context)

        instance = cls(base,
                       show_tariffs_column=True,
                       show_time_icons=False,
                       show_company_icon=search_type == 'plane',
                       show_time_left=True)

        blablacar = instance.blablacar_line(pop_blablacar_row(segments))

        lines = [instance.line(segment) for segment in segments]

        if blablacar:
            lines += [blablacar]

        return lines

    def __json__(self):
        not_found_message = self.not_found_message()

        js_params = {
            'not_found_message': not_found_message,
            'search_type': self.search_type
        }

        js_params.update(self.js_params)

        mods = {}
        if self.is_ajax:
            mods = {'ajax': 'yes'}

            formatted_when = self.form.data['when'].strftime('%Y-%m-%d')

            js_params.update({
                'ajax_info': self.context.ajax_tariffs_info,
                'update-url': langify(reverse('tariffs', kwargs={'search_type': self.search_type})),
                'key': None,
                'when': formatted_when,
                'now': self.context.request.now.strftime('%Y-%m-%d %H:%M')
            })

        if self.mods:
            mods.update(self.mods)

        content = None

        if self.segments:
            content = []

            if self.show_hidden_button():
                content.append({
                    'block': 'b-routers',
                    'elem': 'line',
                    'elemMods': { 'type': 'head-line' },
                    'content': {
                        'elem': 'item',
                        'elemMods': {'type': 'action'},
                        'content': {
                            'block': 'b-action-button',
                            'js': True,
                            'mods': {'type': 'show-hidden'},
                            'mix': {'block': 'b-routers', 'elem': 'action'},
                            'content': gettext('Показать ушедшие'),
                        }
                    }

                })

            json_blablacar_line = self.blablacar_line(pop_blablacar_row(self.segments))

            json_lines = [self.line(segment) for segment in self.segments]
            if json_blablacar_line:
                showed_lines_count = 0
                blablacar_position = -1
                for i, json_line in enumerate(json_lines):
                    if not json_line['hidden']:
                        showed_lines_count += 1

                    if showed_lines_count == BLABLACAR_POSITION:
                        blablacar_position = i + 1
                        break
                if blablacar_position == -1 and json_lines:
                    blablacar_position = len(json_lines)

                if blablacar_position != -1:
                    json_lines.insert(blablacar_position, json_blablacar_line)

            content = content + json_lines

        elif self.search_type in ['plane', 'suburban', 'all'] + TransportType.WATER_TTYPE_CODES:
            content = [ErrorDisclaimerBlock(not_found_message)]

        return {
            'block': 'b-routers',
            'mods': mods,
            'mix': self.mix,
            'js': js_params,
            'content': content
        }

    def show_hidden_button(self):
        if (self.has_gone and not self.has_actual) or \
                (not self.has_gone and self.has_actual) or \
                (not self.has_gone and not self.has_actual):
            return False

        return True


class TimetableTemplate(Base):
    show_calendar = True
    is_plane_page = False
    timetable_block_class = TimetableBlock

    def __init__(self, request, view_context):
        super(TimetableTemplate, self).__init__(request, view_context)

        self.segments = self.context.segments

        self.form = self.context.form
        self.ttype = self.form.ttype

        self.has_gone = any([s.gone for s in self.segments])
        self.has_actual = any([not s.gone for s in self.segments])

        self.show_gone = self.has_gone and not self.has_actual
        self.show_only_aeroexpress = 'aeroex' in self.context.request.GET and \
            set([get_attr_by_path(segment, 'thread.is_aeroexpress', False)
                 for segment in self.segments]) == {True, False}

    def timetable_title(self):
        pass

    def timetable_subtitle(self):
        return {
            'elem': 'subtitle',
            'content': ''
        }

    def b_routers(self):
        t_block = self.timetable_block_class(self,
                                             show_tariffs_column=True,
                                             show_time_icons=self.is_plane_page,
                                             show_company_icon=self.is_plane_page,
                                             show_time_left=True)

        return t_block

    def extended_notice(self):
        if self.context.segments and self.context.extended:
            message = xgettext('Прямых рейсов <request-direction/> не найдено. Предлагаем вам результат поиска <display-direction/>.',
                               request_direction="%s — %s" % (self.context.requested_from.L_title_with_prefix(), self.context.requested_to.L_title_with_prefix()),
                               display_direction="%s — %s" % (self.context.point_from.L_title_with_prefix(), self.context.point_to.L_title_with_prefix()),)

            return ErrorDisclaimerBlock(message)

    def content(self):
        form_renderer = SearchFormRenderer(self.form, hidden=True)

        return [
            self.alarm(),
            {
                'elem': 'item',
                'elemMods': {'type': 'form-container'},
                'content': [
                    {
                        'block': 'b-transport-head',
                        'content': [{
                            'elem': 'title',
                            'content': self.timetable_title()
                        }]
                    },
                    form_renderer.collapsed({'type': 'underhead'}, True, board_route_js_params=self.get_board_route_js_params()),
                    form_renderer,

                ]
            },
            self.body()
        ]

    def get_board_route_js_params(self):
        return True

    def body(self):
        return [{
            'elem': 'item',
            'content': [
                {
                    'block': 'error-message',
                    'content': ''
                },
                self.extended_notice(),
                DetailsAboutLocalTime()
            ]
        },
            self.b_routers(),
            self.context.get('show_avia_disclaimer') and AviaDisclaimer(self.context.request.NATIONAL_VERSION)
        ]


class SuburbanTemplate(TimetableTemplate):
    page = 'search-result-electric-train'

    def head_content_extended(self):
        return [{'elem': 'meta', 'attrs': {'name': 'apple-itunes-app', 'content': 'app-id=387272416'}}]

    def timetable_title(self):
        return gettext('Электрички')

    def timetable_subtitle(self):
        pass

    def get_board_route_js_params(self):
        init_filter_values = {}

        #TODO обобщить механизм фильтров
        if self.show_only_aeroexpress:
            init_filter_values['is-aeroex'] = 'yes'

        return {
            'initFilterValues': init_filter_values
        }


class BusTemplate(TimetableTemplate):
    page = 'search-result-bus'

    def timetable_title(self):
        return gettext('Автобусы')


class TrainTemplate(TimetableTemplate):
    page = 'search-result-train'

    def timetable_title(self):
        return gettext('Поезда')


class PlaneTemplate(TimetableTemplate):
    page = 'search-result-avia'
    is_plane_page = True

    def timetable_title(self):
        return gettext('Самолёты')


class HelicopterTemplate(TimetableTemplate):
    page = 'search-result-train'

    def timetable_title(self):
        return gettext('Вертолёты')


class WaterTemplate(TimetableTemplate):
    page = 'search-result-water'

    def timetable_title(self):
        return gettext('Теплоходы')
